package ru.yandex.direct.jobs.autooverdraft;

import java.time.Duration;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Выгрузка клиентов, которым доступен "Порог отключения" (автоовердрафт), но которые его не включили.
 * Используется для включения им тизера через Я.Аудитории
 */
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 7),
        tags = {DIRECT_PRIORITY_1, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class AutooverdraftClietnsForTeaserJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(AutooverdraftClietnsForTeaserJob.class);

    private static final String QUERY_RESOURCE = "classpath:///autooverdraft/autooverdfart_clients_for_teaser.yql";
    static final String EXPORT_PATH = "export/autooverdraft_clients_for_teaser";


    private static final Duration GENERATION_INTERVAL = Duration.ofHours(12);

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;


    @Autowired
    public AutooverdraftClietnsForTeaserJob(DirectExportYtClustersParametersSource parametersSource,
                                            YtProvider ytProvider) {
        this.parametersSource = parametersSource;

        exportBuilder =
                new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                        .withBindingsProvider(AutooverdraftClietnsForTeaserJob::getBindings);
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded(GENERATION_INTERVAL);
    }

    private static Object[] getBindings(CommonYqlExport.Context ctx) {

        Set<String> setOfClients = new HashSet<>();
        for (long item : AutoOverdraftUtils.CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS) {
            setOfClients.add(String.valueOf(item));
        }
        String badClients = String.join(", ", setOfClients);

        Set<String> setOfCurrencies = new HashSet<>();
        for (CurrencyCode currencyCode : AutoOverdraftUtils.AVAILABLE_CURRENCY_CODES) {
            setOfCurrencies.add(currencyCode.name());
        }
        String availableCurrencies = String.join(", ", setOfCurrencies);

        return new Object[]{
                ctx.getDestinationTablePathAsString(),
                badClients,
                availableCurrencies,
        };
    }
}
