package ru.yandex.direct.jobs.autooverdraft;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.autooverdraft.model.ClientOverdraftLimitChanges;
import ru.yandex.direct.core.entity.autooverdraftmail.repository.ClientOverdraftLimitChangesRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyAmount;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Translator;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderTemplateParams;

import static ru.yandex.direct.common.configuration.CommonConfiguration.DIRECT_EXECUTOR_SERVICE;

/**
 * Сервис отправки писем о пересчёте лимита овердрафта
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-125273">
 *     DIRECT-125273: Отправлять письма про смену лимита овердрафта</a>
 * Шаблон в тестинге
 * https://test.sender.yandex-team.ru/direct/campaign/18921/overview
 * Шаблон в проде:
 * https://sender.yandex-team.ru/direct/campaign/293714/overview
 */
@ParametersAreNonnullByDefault
@Service
public class OverdraftLimitChangesMailSenderService {
    private static final Locale RU = new Locale.Builder().setLanguageTag("ru").build();
    private static final Translator RU_TRANSLATOR = I18NBundle.makeStubTranslatorFactory().getTranslator(RU);
    private static final String LOGIN = "login";
    private static final String OVERDRAFT_LIMIT = "overdraft_limit";
    private static final String CLIENT_ID = "ClientID";
    private final YandexSenderClient senderClient;
    private final String slug;
    private final ShardHelper shardHelper;
    private final ClientOverdraftLimitChangesRepository repository;
    private final ExecutorService executorService;

    @Autowired
    public OverdraftLimitChangesMailSenderService(DirectConfig config,
                                                  YandexSenderClient senderClient,
                                                  ShardHelper shardHelper,
                                                  ClientOverdraftLimitChangesRepository repository,
                                                  @Qualifier(DIRECT_EXECUTOR_SERVICE) ExecutorService executorService) {
        this.senderClient = senderClient;
        slug = config.getBranch("auto_overdraft_mails").getString("OVERDRAFT_RECALCULATED");
        this.shardHelper = shardHelper;
        this.repository = repository;
        this.executorService = executorService;
    }

    boolean sendOverdraftLimitChangedEmail(User user, CurrencyAmount amount) {
        String login = user.getLogin();
        String clientId = user.getClientId().toString();
        String overdraftLimit = amount.translate(RU_TRANSLATOR);
        // в шаблоне после лимита стоит точка. По правилам точка от аббревиатуры должна её съедать.
        overdraftLimit = overdraftLimit.replaceAll("\\.$", "");
        String email = user.getRecommendationsEmail() != null ? user.getRecommendationsEmail() : user.getEmail();
        var params = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(slug)
                .withToEmail(email)
                .withAsync(true)
                .withArgs(ImmutableMap.of(LOGIN, login, OVERDRAFT_LIMIT, overdraftLimit, CLIENT_ID, clientId))
                .build();
        return senderClient.sendTemplate(params, YandexSenderClient::isInvalidToEmail);
    }

    void addAsSent(ClientId clientId) {
        var shard = shardHelper.getShardByClientId(clientId);
        repository.add(shard, clientId, true);
    }

    /**
     * Очистить таблицу с клиентами для отправки писем от тех, кому уже отправили письмо
     *
     * @return количество удалённых записей
     */
    int deleteSentInMailTable() {
        var clientIds = getClientsToSendMail().get(true);
        if (clientIds == null) {
            return 0;
        }
        return deleteFromMailTableByClientIds(clientIds);
    }

    /**
     * Очистить таблицу с клиентами, id которых передали
     *
     * @param clientIds — список id клиентов
     * @return количество удалённых записей
     */
    int deleteFromMailTableByClientIds(List<ClientId> clientIds) {
        var count = new AtomicInteger(0);
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .chunkedBy(1000)
                .forEach((shard, clientIdsChunk) ->
                        count.getAndAdd(repository.deleteByClientIds(shard, clientIdsChunk)));
        return count.get();
    }

    /**
     * Получить списки тех, кому нужно отправить письмо об изменении лимитА, и тех, кому не нужно
     *
     * @return словарь из Boolean в списки id клиентов. В false те, кому нужно отправить письмо, в true те, кому не надо
     */
    public Map<Boolean, List<ClientId>> getClientsToSendMail() {
        return shardHelper.forEachShardParallel(repository::getAll, executorService)
                .values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.groupingBy(ClientOverdraftLimitChanges::getIsNotificationSent,
                        Collectors.mapping(ClientOverdraftLimitChanges::getClientId, Collectors.toList())));
    }
}

