package ru.yandex.direct.jobs.autooverdraft;

import java.math.BigDecimal;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyAmount;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;
import ru.yandex.yql.YqlPreparedStatement;

import static com.google.common.collect.Lists.partition;
import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion.SQLv1;

/**
 * Собирает список клиентов, у которых поменялся лимит овердрафта и отправляет им письма с новыми правильными значениями
 * Для отправки письма овердрафт должен стать доступен в Балансе (таких клиентов мы собираем в течение дня),
 * или лимит должен поменяться с ненулевого на другой ненулевой в таблице-источнике в YT
 * и либо у клиента овердрафт не подключен, либо лимит уменьшился ниже поставленного клиентом
 *
 * Во вторую очередь собираем клиентов из специальной таблицы, заполняемой из интапи, довыбираем им данные и тоже
 * шлём им письма.
 */
/*
TODO: DIRECT-161137 Если решим включать обратно, то надо будет перенести документацию
https://wiki.yandex-team.ru/users/sco76/overdraftlimitchangesmailerjob/ в doc и передать в аппдьюти
@Hourglass(cronExpression = "0 0 10,11,12 * * ?", needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS},
        notifications = {@OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT},
                method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class)
 */
@ParametersAreNonnullByDefault
public class OverdraftLimitChangesMailerJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(OverdraftLimitChangesMailerJob.class);
    private static final String PATH_PREFIX = "//home/mlp/prod/overdraft_limits_direct/";
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("autooverdraft/overdraft_changes.yql").readLines());
    private static final long YQL_QUERY_TIMEOUT_MINUTES = 15L;

    private final YtProvider ytProvider;
    private final PpcPropertiesSupport propertiesSupport;
    private final OverdraftLimitChangesMailSenderService senderService;
    private final ClientService clientService;
    private final ClientNdsService clientNdsService;
    private final UserService userService;
    private final Map<ClientId, User> clientChefs = new HashMap<>();
    private final Map<ClientId, Percent> clientNds = new HashMap<>();
    final Map<ClientId, OverdraftLimitChangesInfo> changesToSend = new HashMap<>();

    @Autowired
    public OverdraftLimitChangesMailerJob(YtProvider ytProvider,
                                          PpcPropertiesSupport propertiesSupport,
                                          OverdraftLimitChangesMailSenderService senderService,
                                          ClientService clientService,
                                          ClientNdsService clientNdsService,
                                          UserService userService) {
        this.ytProvider = ytProvider;
        this.propertiesSupport = propertiesSupport;
        this.senderService = senderService;
        this.clientService = clientService;
        this.clientNdsService = clientNdsService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        var property = propertiesSupport.get(PpcPropertyNames.OVERDRAFT_LIMIT_MAILER_LAST_TABLE_NAME);
        String lastTableName = property.get();
        if (lastTableName == null) {
            logger.error("Property {} is not set", PpcPropertyNames.OVERDRAFT_LIMIT_MAILER_LAST_TABLE_NAME);
            setJugglerStatus(JugglerStatus.WARN, "property is not set");
            return;
        }
        String todaysTableName = LocalDate.now().format(DateTimeFormatter.ISO_DATE);
        if (todaysTableName.equals(lastTableName)) {
            logger.info("Table {} is already processed", todaysTableName);
            return;
        }

        if (!processYtTable(lastTableName, todaysTableName)) {
            return; // произошла ошибка, мы её уже зажурналировали
        }
        logger.info("To send after processing YT table: {}", changesToSend.size());
        processPpcMailTable();
        logger.info("To send total: {}", changesToSend.size());

        var count = sendEmails();
        logger.info("Sent {} emails", count);

        property.set(todaysTableName);
        count = senderService.deleteSentInMailTable();
        logger.info("Deleted {} records from mail table", count);
    }

    private boolean processYtTable(String lastTableName, String todaysTableName) {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER);
        var todaysTable = new YtTable(PATH_PREFIX + todaysTableName);
        if (!ytOperator.exists(todaysTable)) {
            logger.error("Table {}{} does not exist", PATH_PREFIX, todaysTableName);
            setJugglerStatus(JugglerStatus.CRIT, "today's table does not exist");
            return false;
        }
        if (!ytOperator.exists(new YtTable(PATH_PREFIX + lastTableName))) {
            logger.error("Table {}{} does not exist", PATH_PREFIX, lastTableName);
            setJugglerStatus(JugglerStatus.CRIT, "previous table does not exist");
            return false;
        }

        YPath changesTablePath;
        try (TraceProfile ignore = Trace.current().profile("overdraft_changes:yql")) {
            changesTablePath = createChangesTable(PATH_PREFIX + lastTableName, PATH_PREFIX + todaysTableName);
        } catch (SQLException | ExecutionException | TimeoutException e) {
            logger.error("Got an YT query exception", e);
            setJugglerStatus(JugglerStatus.WARN, "YT query was not successful");
            return false;
        }

        var changesTable = new YtTable(changesTablePath.toString());
        if (ytOperator.readTableRowCount(changesTable) == 0) {
            logger.info("No changes today");
        } else {
            var tableRow = new OverdraftLimitChangesTableRow();
            var count = ytOperator.readTableSnapshot(changesTable,
                    tableRow,
                    this::convertRow,
                    this::processRows,
                    1000);
            logger.info("Processed {} changes to mail", count);
        }
        return true;
    }

    void processPpcMailTable() {
        var clientsToSendMap = senderService.getClientsToSendMail();
        var clientIdsNotToSend = clientsToSendMap.get(true);
        if (clientIdsNotToSend != null && !clientIdsNotToSend.isEmpty()) {
            changesToSend.keySet().removeAll(clientIdsNotToSend);
        }
        var clientIdsToSend = clientsToSendMap.get(false);
        if (clientIdsToSend == null) {
            return;
        }

        var clientIdsToDeleteFromTable = new ArrayList<ClientId>();
        for (var clientIdsToSendChunk : partition(clientIdsToSend, 1000)) {
            processPpcChunk(clientIdsToSendChunk, clientIdsToDeleteFromTable);
        }
        if (!clientIdsToDeleteFromTable.isEmpty()) {
            var count = senderService.deleteFromMailTableByClientIds(clientIdsToDeleteFromTable);
            logger.info("Deleted {} bad records from mail table", count);
        }
    }

    private void processPpcChunk(List<ClientId> clientIdsToSendChunk, List<ClientId> clientIdsToDeleteFromTable) {
        var filteredClientIdsChunk = clientIdsToSendChunk.stream()
                .filter(id -> !changesToSend.containsKey(id)).collect(Collectors.toList());
        var clientsMap = clientService.massGetClientsByClientIds(filteredClientIdsChunk);
        var clientIdsToFetchChefs = new ArrayList<ClientId>();
        var clientsToFetchVat = new ArrayList<Client>();
        for (ClientId clientId : filteredClientIdsChunk) {
            if (!clientsMap.containsKey(clientId)) {
                logger.error("Client {} was not found", clientId); // не должно случаться, клиенты не удаляются
                clientIdsToDeleteFromTable.add(clientId);
                continue;
            }
            Client client = clientsMap.get(clientId);
            if (isOverdraftAvailableForClient(client)) {
                var changesInfo = new OverdraftLimitChangesInfo(clientId,
                        client.getOverdraftLimit().doubleValue(),
                        client.getWorkCurrency().toString(),
                        false);
                changesToSend.put(clientId, changesInfo);
                clientIdsToFetchChefs.add(clientId);
                clientsToFetchVat.add(client);
            } else {
                logger.info("Client {} no longer has access to overdraft limit", clientId);
                clientIdsToDeleteFromTable.add(clientId);
            }
        }
        fetchExtraDataFromDb(clientIdsToFetchChefs, clientsToFetchVat);
    }

    boolean isOverdraftAvailableForClient(Client client) {
        var overdraftLimit = client.getOverdraftLimit();
        var isBalanceBanned = client.getStatusBalanceBanned();
        return (overdraftLimit.compareTo(BigDecimal.ZERO) > 0)
                && !isBalanceBanned.equals(Boolean.TRUE);
    }

    private long sendEmails() {
        long count = 0L;
        for (var entry : changesToSend.entrySet()) {
            var clientId = entry.getKey();
            count += sendEmail(clientId, entry.getValue());
            senderService.addAsSent(clientId);
        }

        return count;
    }

    private int sendEmail(ClientId clientId, OverdraftLimitChangesInfo changesInfo) {
        if (!clientChefs.containsKey(clientId)) {
            logger.error("Chief user for client {} was not found", clientId);
            return 0;
        }

        if (!clientNds.containsKey(clientId)) {
            logger.error("VAT was not found for client {}", clientId);
            return 0;
        }

        var moneyAmount = Money.valueOf(changesInfo.getOverdraftLimit(),
                CurrencyCode.valueOf(changesInfo.getWorkCurrency()));
        var currencyAmount = CurrencyAmount.fromMoney(moneyAmount.subtractNds(clientNds.get(clientId)));
        logger.info("Sending an email to client {}, new limit: {} {}",
                clientId, changesInfo.getOverdraftLimit(), changesInfo.getWorkCurrency());
        try {
            if (senderService.sendOverdraftLimitChangedEmail(clientChefs.get(clientId), currencyAmount)) {
                return 1;
            } else {
                logger.error("Couldn't send an email to client {}, bad email address", clientId);
            }
        } catch (YandexSenderException e) {
            logger.error("Couldn't send an email to client", e);
        }
        return 0;
    }

    private OverdraftLimitChangesInfo convertRow(OverdraftLimitChangesTableRow row) {
        return new OverdraftLimitChangesInfo(row.getClientId(), row.getOverdraftLimit(), row.getWorkCurrency(),
                row.isNewLimitSmaller());
    }

    private void processRows(List<OverdraftLimitChangesInfo> overdraftLimitChangesInfos) {
        var clientIds = overdraftLimitChangesInfos.stream()
                .filter(i -> CurrencyCode.isRealCurrencyCode(i.getWorkCurrency()))
                .map(OverdraftLimitChangesInfo::getClientId)
                .collect(Collectors.toList());
        var clientsMap = clientService.massGetClientsByClientIds(clientIds);
        var clientIdsToFetchChefs = new ArrayList<ClientId>();
        var clientsToFetchVat = new ArrayList<Client>();
        for (OverdraftLimitChangesInfo info : overdraftLimitChangesInfos) {
            if (!CurrencyCode.isRealCurrencyCode(info.getWorkCurrency())) {
                logger.warn("Client {} has unknown currency {}", info.getClientId(), info.getWorkCurrency());
            } else if (!clientsMap.containsKey(info.getClientId())) {
                logger.warn("Client {} was not found", info.getClientId());
            } else {
                processClient(clientsMap.get(info.getClientId()), info, clientIdsToFetchChefs, clientsToFetchVat);
            }
        }

        fetchExtraDataFromDb(clientIdsToFetchChefs, clientsToFetchVat);
    }

    // предполагается, что оба списка заполняются синхронно
    void fetchExtraDataFromDb(List<ClientId> clientIdsToFetchChefs, List<Client> clientsToFetchVat) {
        if (!clientIdsToFetchChefs.isEmpty()) {
            clientChefs.putAll(userService.getChiefUserByClientIdMap(clientIdsToFetchChefs));
            clientNds.putAll(clientNdsService.massGetEffectiveClientNds(clientsToFetchVat).stream()
                    .collect(Collectors.toMap(n -> ClientId.fromLong(n.getClientId()), ClientNds::getNds)));
        }
    }

    void processClient(Client client, OverdraftLimitChangesInfo info, List<ClientId> clientIdsToFetchChefs,
                               List<Client> clientsToFetchVat) {
        if (client.getWorkCurrency() != CurrencyCode.valueOf(info.getWorkCurrency())) {
            logger.warn("Client {} has different currency, expected {}, got {}",
                    info.getClientId(), client.getWorkCurrency(), info.getWorkCurrency());
            return;
        }

        var autoOverdraftLimit = client.getAutoOverdraftLimit();
        if (isOverdraftAvailableForClient(client)
                && (autoOverdraftLimit.compareTo(BigDecimal.ZERO) == 0
                    || (autoOverdraftLimit.compareTo(BigDecimal.valueOf(info.getOverdraftLimit())) > 0
                    && info.isNewLimitSmaller()))) {
            changesToSend.put(info.getClientId(), info);
            clientIdsToFetchChefs.add(info.getClientId());
            clientsToFetchVat.add(client);
        }
    }

    private YPath createChangesTable(String lastTableName, String todaysTableName)
            throws SQLException, ExecutionException, TimeoutException {
        String changesTableName = YtPathUtil.generateTemporaryPath();
        try (YqlConnection connection = (YqlConnection) ytProvider.getYql(YT_CLUSTER, SQLv1).getConnection();
             YqlPreparedStatement statement = (YqlPreparedStatement) connection.prepareStatement(YQL_QUERY)) {
            statement.setString(1, lastTableName);
            statement.setString(2, todaysTableName);
            statement.setString(3, changesTableName);
            statement.setAcl(ytProvider.getClusterConfig(YT_CLUSTER).getDefaultYqlAcl());
            ResultSetFuture future = (ResultSetFuture) statement.beginExecuteQuery();
            logger.info("Started YQL execution, OperationID: {}", future.getOperationId());
            future.get(YQL_QUERY_TIMEOUT_MINUTES, MINUTES);
            logger.info("Finished YQL execution, temp dir path: {}", changesTableName);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return YPath.simple(changesTableName);
    }
}
