package ru.yandex.direct.jobs.balance;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.balance.repository.BalanceInfoQueueRepository;
import ru.yandex.direct.dbschema.ppc.tables.BalanceInfoQueue;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;

/**
 * Удаление из таблицы {@link ru.yandex.direct.dbschema.ppc.tables.BalanceInfoQueue}
 * записей со статусом отправки status_send = "Error" старше {@code LIMIT_AGE_DAYS} дней
 * и записей со статусом "Send"
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 24),
        //PRIORITY: под вопросом; лимит 7 дней и пользователям это не видно - !!уточнить у @ppalex!!
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 7200)
public class BalanceInfoQueueCleaner extends DirectShardedJob {

    private static final String CLEAR_BALANCE_INFO_QUEUE_PROPERTY = "CLEAR_BALANCE_INFO_QUEUE_SHARD_%d";

    private static final int DELETE_CHUNK_SIZE = 10_000;
    private static final int SELECT_CHUNK_SIZE = 100_000;
    private static final int LIMIT_AGE_DAYS = 7;

    private static final Logger logger = LoggerFactory.getLogger(BalanceInfoQueue.class);

    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(5.0);
    private final BalanceInfoQueueRepository balanceInfoQueueRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public BalanceInfoQueueCleaner(BalanceInfoQueueRepository balanceInfoQueueRepository,
                                   PpcPropertiesSupport ppcPropertiesSupport,
                                   SolomonPushClient solomonPushClient) {
        this.balanceInfoQueueRepository = balanceInfoQueueRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.solomonPushClient = solomonPushClient;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    BalanceInfoQueueCleaner(int shard, BalanceInfoQueueRepository balanceInfoQueueRepository,
                            PpcPropertiesSupport ppcPropertiesSupport,
                            SolomonPushClient solomonPushClient) {
        super(shard);
        this.balanceInfoQueueRepository = balanceInfoQueueRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        PpcProperty<LocalDate> property = ppcPropertiesSupport.get(getPropertyName());

        if (isRunningWithinCurrentDay()) {
            logger.debug("Already worked today ({})", property.get());
            return;
        }

        LocalDateTime boundaryDate = LocalDateTime.now().minusDays(LIMIT_AGE_DAYS);

        logger.debug("get records with send_status = Error (add_time before {}) or send_status = Send (add_time = any)",
                boundaryDate);

        int selectedRowsCount;
        int deletedRowsTotal = 0;
        do {
            List<Long> errorsAndSentOutdatedIds = balanceInfoQueueRepository
                    .getSentOrOutdatedErrorIds(getShard(), boundaryDate, limited(SELECT_CHUNK_SIZE));
            selectedRowsCount = errorsAndSentOutdatedIds.size();
            logger.debug("selected {} rows", selectedRowsCount);

            for (List<Long> idsToDeleteChunk : Iterables.partition(errorsAndSentOutdatedIds, DELETE_CHUNK_SIZE)) {
                logger.debug("delete next records chunk");
                logger.trace("Ids for delete: {}", idsToDeleteChunk);
                deletedRowsTotal += relaxedWorker.callAndRelax(() -> {
                    int deletedRows = balanceInfoQueueRepository.deleteByIds(getShard(), idsToDeleteChunk);
                    logger.info("deleted {} records", deletedRows);
                    return deletedRows;
                });
            }

        } while (selectedRowsCount == SELECT_CHUNK_SIZE);

        property.set(LocalDate.now());

        var registry = SolomonUtils.newPushRegistry("flow", "balance_info_queue");
        registry.gaugeInt64("deleted_rows_count", Labels.of("shard", String.valueOf(getShard())))
                .set(deletedRowsTotal);
        solomonPushClient.sendMetrics(registry);
    }

    PpcPropertyName<LocalDate> getPropertyName() {
        return new PpcPropertyName<>(String.format(CLEAR_BALANCE_INFO_QUEUE_PROPERTY, getShard()),
                PpcPropertyType.LOCAL_DATE);
    }


    /**
     * Проверка на запуск в течение последних суток.
     *
     * @return true если джоба запускалась в течение последних суток. Иначе false
     */
    boolean isRunningWithinCurrentDay() {
        return LocalDate.now().equals(ppcPropertiesSupport.get(getPropertyName()).get());
    }
}
