package ru.yandex.direct.jobs.balance.billingaggregatesexport;

import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.common.db.PpcPropertyNames.BILLING_AGGREGATES_FORCE_FULL_EXPORT_ONCE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Экспорт cid и wallet_cid всех биллинговых агрегатов в export/billing_aggregates
 * Таблица должна обновляться каждый день до 6 утра и содержать данные актуальные на конец предыдущего дня.
 * <p>
 * Джоба делает инкрементальный апдейт каждый день и полную переналивку каждый месяц.
 * Таблица может содержать биллинговые агрегаты, которые были удалены в Директе, но это нормально.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_DIRECT_BILL_AGG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_0})
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class BillingAggregatesExportJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(BillingAggregatesExportJob.class);
    private static final int INCREMENTAL_UPLOAD_OVERLAP_DAYS = 1;

    @Autowired
    private YtProvider ytProvider;

    @Autowired
    private DslContextProvider dslContextProvider;

    @Autowired
    private ShardSupport shardSupport;

    @Autowired
    private PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    private DirectExportYtClustersParametersSource parametersSource;

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        PpcProperty<Boolean> forceReuploadProp = ppcPropertiesSupport.get(BILLING_AGGREGATES_FORCE_FULL_EXPORT_ONCE);
        ExportState state = ExportState.determine(ytProvider, ytCluster, forceReuploadProp);
        BillingAggregatesUploader uploader = new BillingAggregatesUploader(
                ytProvider,
                dslContextProvider,
                shardSupport,
                ytCluster
        );

        if (!state.doesExportTableExist() || state.needFullReupload()) {
            logger.info("Reuploading full table");
            uploader.reuploadFullTable(state);
        } else if (state.needIncrementalUpdate()) {
            LocalDateTime incrementSince = state.getExportTableUploadTime().minusDays(INCREMENTAL_UPLOAD_OVERLAP_DAYS);
            logger.info("Making incremental merge since {}", incrementSince);
            uploader.doIncrementalMerge(state, incrementSince);
        }

        if (state.isForceReuploadRequested()) {
            logger.info("Resetting forced reupload flag.");
            forceReuploadProp.set(false);
        }
    }
}
