package ru.yandex.direct.jobs.balance.billingaggregatesexport;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.specs.MergeMode;
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

/**
 * Чтение биллинговых агрегатов из базы и загрузка их в статические YT таблицы.
 * <p>
 * Может производить инкрементальное обновление.
 */
class BillingAggregatesUploader {
    private static final Logger logger = LoggerFactory.getLogger(BillingAggregatesUploader.class);

    static final String EXPORT_PATH = "export/billing_aggregates";
    /**
     * Директория для промежуточных результатов.
     * <p>
     * !!! Будет удалена целиком после успешного экспорта.
     */
    private static final String TMP_PATH = "tmp/billing_aggregates";

    private final Yt yt;
    private final DslContextProvider dslContextProvider;
    private final int shardCount;
    private final YPath tmpDir;
    private final YtOperator ytOperator;
    private final YPath ytExportPath;

    /**
     * Схема для временных статических таблиц.
     * Отличается схемы финальной таблицы тем, что допускает неуникальные ключи.
     * Они могут возникнуть в случае, если клиента решардировали, и два агрегата с одним Id оказались в разных шардах.
     */
    private static final TableSchema tempSchema = new TableSchema.Builder()
            .addValue("cid", ColumnValueType.INT64)
            .addValue("wallet_cid", ColumnValueType.INT64)
            .setUniqueKeys(false)
            .build();

    /**
     * Схема для результирующих таблиц.
     */
    private static final TableSchema outputSchema = new TableSchema.Builder()
            .addKey("cid", ColumnValueType.INT64)
            .addValue("wallet_cid", ColumnValueType.INT64)
            .build();

    BillingAggregatesUploader(YtProvider ytProvider,
                              DslContextProvider dslContextProvider, ShardSupport shardSupport, YtCluster ytCluster) {
        this.yt = ytProvider.get(ytCluster);
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(ytCluster);
        this.dslContextProvider = dslContextProvider;
        this.shardCount = shardSupport.getAvailablePpcShards().size();
        this.tmpDir = YPath.simple(
                YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), TMP_PATH)
        );
        this.ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);
        this.ytExportPath = YPath.simple(
                YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), EXPORT_PATH)
        );
    }


    /**
     * Полная перезаливка биллинговых агрегатов.
     */
    void reuploadFullTable(ExportState exportState) {
        YPath newTable = tmpDir.child("new_table");
        createYtTable(newTable, tempSchema);
        logger.info("Writing all billing aggregates into {}", newTable);
        writeYtTableFromIterator(newTable, new BillingAggregatesIterator(dslContextProvider, shardCount, null));

        // Избавляемся от дубликатов биллинговых агрегатов
        YPath groupedTable = tmpDir.child("grouped_table");
        logger.info("Grouping billing aggregates using YQL into {}", groupedTable);
        ytOperator.yqlExecute(
                "$groupedTable = ?;\n"
                        + "$newTable = ?;\n"
                        + "INSERT INTO $groupedTable WITH TRUNCATE\n"
                        + "SELECT nt.cid AS cid, MAX(nt.wallet_cid) AS wallet_cid\n"
                        + "FROM $newTable AS nt\n"
                        + "GROUP BY nt.cid\n"
                        + "ORDER BY cid\n",
                groupedTable.toString(),
                newTable.toString()
        );

        // В groupedTable в общем-то уже лежит готовая таблица, но в её схеме не выставлен атрибут unique_keys: true
        // С помощью merge приводим схему в порядок.
        mergeAndExport(exportState, groupedTable);

        yt.cypress().remove(tmpDir);
    }

    /**
     * Инкрементальное обновление.
     * Из базы вычитываются биллинговые агрегаты, созданные после {@code incrementSince}
     * и примерживаются к экспортной таблице.
     */
    void doIncrementalMerge(ExportState state, LocalDateTime incrementSince) {
        List<Row> allNew = getNewBillingAggregates(incrementSince);
        if (allNew.isEmpty()) {
            return;
        }
        Row minRow = Collections.min(allNew, Comparator.comparingLong(row -> row.cid));
        logger.info("New row min = {}", minRow.cid);

        // Читаем из экспортной таблицы все строки, которые могут пересекаться со списком allNew
        Set<Long> existingIds = getExistingBillingAggregateIds(minRow.cid);

        // Группируем строки по cid и отфильтровываем те, которые уже есть в экспортной таблице.
        // Важно, чтобы результирующий список был отсортирован по cid, иначе заливка в YT упадёт.
        List<Row> newList = EntryStream.of(allNew)
                .mapKeys(i -> allNew.get(i).cid)
                .collapseKeys((a, b) -> a.walletCid > b.walletCid ? a : b)
                .filterKeys(cid -> !existingIds.contains(cid))
                .values()
                .sortedByLong(row -> row.cid)
                .toList();
        if (newList.isEmpty()) {
            logger.info("No new billing aggregates found");
            return;
        }

        YPath outputTable = tmpDir.child("output_table");
        createYtTable(outputTable, outputSchema);
        logger.info("Writing new billing aggregates into {}", outputTable);
        writeYtTableFromIterator(outputTable, newList.iterator());

        mergeAndExport(state, ytExportPath, outputTable);

        yt.cypress().remove(tmpDir);
    }

    /**
     * Берет одну или несколько таблиц с путями {@code paths} мержит их,
     * выставляет атрибут с временем заливки и заменяет ей таблицу экспорта.
     */
    private void mergeAndExport(ExportState exportState, YPath... paths) {
        YtTable newExportTable = new YtTable(tmpDir.child("new_export").toString());
        createYtTable(newExportTable.ypath(), outputSchema);
        MergeSpec mergeSpec = MergeSpec.builder()
                .setMergeMode(MergeMode.SORTED)
                .setCombineChunks(true)
                .setMergeBy(Collections.singletonList("cid"))
                .setInputTables(Arrays.asList(paths))
                .setOutputTable(newExportTable.ypath())
                .build();
        yt.operations().mergeAndGetOp(mergeSpec).awaitAndThrowIfNotSuccess();
        ytOperator.writeTableUploadTime(newExportTable, exportState.getNewUploadTime());
        yt.cypress().move(newExportTable.ypath(), ytExportPath, true);
    }

    /**
     * Читает из экспортной таблицы все строки, у которых cid >= переданному параметру.
     */
    private Set<Long> getExistingBillingAggregateIds(long cid) {
        YPath exportRange = ytExportPath.withRange(
                new RangeLimit(Cf.list(YTree.integerNode(cid)), -1, -1),
                new RangeLimit(Cf.list(), -1, -1)
        );
        logger.info("Reading existing billing aggregates from {}", exportRange);
        Set<Long> existingCids = new HashSet<>();
        yt.tables().read(
                exportRange,
                YTableEntryTypes.yson(Row.class),
                row -> {
                    existingCids.add(row.cid);
                }
        );
        logger.info("Read {} existing cids", existingCids.size());

        return existingCids;
    }

    /**
     * Читает из всех шардов БД все биллинговые агрегаты созданные после {@code incrementSince}.
     * <p>
     * Возвращает список строк, готовых к записи в YT.
     */
    private List<Row> getNewBillingAggregates(LocalDateTime incrementSince) {
        BillingAggregatesIterator iterator =
                new BillingAggregatesIterator(dslContextProvider, shardCount, incrementSince);
        List<Row> rows = new ArrayList<>();
        logger.info("Looking for new billing aggregates created after {}", incrementSince);
        while (iterator.hasNext()) {
            rows.add(iterator.next());
        }
        logger.info("Found {} rows", rows.size());

        return rows;
    }

    /**
     * Создание статической таблицы в YT с переданной схемой.
     * Если таблица уже существует, ничего не делается.
     */
    private void createYtTable(YPath ypath, TableSchema schema) {
        Map<String, YTreeNode> attributes = Collections.singletonMap(YtUtils.SCHEMA_ATTR, schema.toYTree());
        yt.cypress().create(
                /* transactionId */ Optional.empty(),
                /* pingAncestorTransactions */ false,
                ypath,
                CypressNodeType.TABLE,
                /* recursive */ true,
                /* ignoreExisting */ true,
                attributes
        );
    }

    /**
     * Записывает в статическую таблицу YT строки с биллинговыми агрегатами из итератора.
     * Если таблица непустая, её содержимое перезаписывается.
     */
    private void writeYtTableFromIterator(YPath ypath, Iterator<Row> iterator) {
        yt.tables().write(Optional.empty(), false, ypath, YTableEntryTypes.yson(Row.class),
                iterator
        );
    }

    /**
     * Итератор, ходящий по всем шардам БД подряд, и возвращающий строки с биллинговыми агрегатами,
     * готовыми для записи в YT.
     * <p>
     * Вычитывает из базы по {@value DEFAULT_CHUNK_SIZE} строк за один поход.
     */
    static class BillingAggregatesIterator implements Iterator<Row> {
        private static final int DEFAULT_CHUNK_SIZE = 10_000;

        private DslContextProvider dslContextProvider;

        @Nullable
        private LocalDateTime createdAfter;

        private int chunkSize;
        private List<Row> nextChunk;
        private int nextIndex;

        private int nextShard = 1;
        private int shardsCount;
        private long minCid = 0L;

        BillingAggregatesIterator(DslContextProvider dslContextProvider,
                                  int shardsCount,
                                  @Nullable LocalDateTime createdAfter) {
            this.nextChunk = Collections.emptyList();
            this.dslContextProvider = dslContextProvider;
            this.shardsCount = shardsCount;
            this.createdAfter = createdAfter;
            this.chunkSize = DEFAULT_CHUNK_SIZE;
        }

        @Override
        public boolean hasNext() {
            if (nextIndex < nextChunk.size()) {
                return true;
            }
            readNextChunk();
            return !nextChunk.isEmpty();
        }

        @Override
        public Row next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            return nextChunk.get(nextIndex++);
        }

        void setChunkSize(int chunkSize) {
            this.chunkSize = chunkSize;
        }

        private void readNextChunk() {
            nextChunk.clear();
            nextIndex = 0;

            while (nextShard <= shardsCount) {
                nextChunk = readRowsFromDb();

                if (nextChunk.size() < chunkSize) {
                    nextShard++;
                    minCid = 0;
                } else {
                    Row lastRow = nextChunk.get(nextChunk.size() - 1);
                    minCid = lastRow.cid;
                }

                if (!nextChunk.isEmpty()) {
                    break;
                }
            }
        }

        private List<Row> readRowsFromDb() {
            Condition createAfterConditon = (
                    createdAfter != null ?
                            CAMP_OPTIONS.CREATE_TIME.ge(createdAfter) :
                            DSL.trueCondition()
            );

            logger.info("Reading from shard {}, after {}", nextShard, minCid);
            List<Row> rows = dslContextProvider.ppc(nextShard)
                    .select(CAMPAIGNS.CID, CAMPAIGNS.WALLET_CID)
                    .from(CAMPAIGNS)
                    .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                    .where(
                            // нет условий на statusEmpty и archived, т.к. у биллинговых агрегатов они всегда No
                            // а также в этой выгрузке удалённые/архивные лишними не будут.
                            CAMPAIGNS.CID.gt(minCid)
                                    .and(CAMPAIGNS.TYPE.eq(CampaignsType.billing_aggregate))
                                    .and(createAfterConditon)
                    )
                    .orderBy(CAMPAIGNS.CID)
                    .limit(chunkSize)
                    .fetch()
                    .map(record -> new Row(record.value1(), record.value2()));
            logger.info("Read {} rows", rows.size());

            return rows;
        }
    }
}
