package ru.yandex.direct.jobs.balance.billingaggregatesexport;

import java.time.LocalDateTime;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.jobs.balance.billingaggregatesexport.BillingAggregatesUploader.EXPORT_PATH;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

/**
 * Контейнер для хранения статуса экспорта в начале работы.
 */
class ExportState {
    private static final Logger logger = LoggerFactory.getLogger(ExportState.class);

    private boolean exportTableExists;
    private LocalDateTime exportTableUploadTime;
    private LocalDateTime newUploadTime;
    private boolean forceReuploadRequested;

    /**
     * Вычисляет состояние экспорта по данным из YT и флагу принудительной полной переналивки.
     */
    static ExportState determine(YtProvider ytProvider, YtCluster ytCluster, PpcProperty<Boolean> forceReuploadProp) {
        ExportState state = new ExportState();

        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(ytCluster);
        YtTable exportTable =
                new YtTable(YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), EXPORT_PATH));

        if (ytOperator.exists(exportTable)) {
            state.exportTableExists = true;
            String updateTime = ytProvider.getOperator(ytCluster).readTableUploadTime(exportTable);
            state.exportTableUploadTime = LocalDateTime.parse(updateTime);

            logger.info("Found export table, update time is {}", state.exportTableUploadTime);
        } else {
            logger.info("No export table found!");
        }

        state.newUploadTime = LocalDateTime.now();

        state.forceReuploadRequested = forceReuploadProp.getOrDefault(false);

        return state;
    }

    /**
     * Существует ли экспортная таблица - значит, что когда-то уже был успешный экспорт.
     */
    boolean doesExportTableExist() {
        return exportTableExists;
    }

    /**
     * Нужно ли сделать полную переналивку таблицы.
     * Делаем это по запросу через {@code ppc_property BILLING_AGGREGATES_FORCE_FULL_EXPORT_ONCE},
     * и в начале каждого месяца для профилактики.
     */
    boolean needFullReupload() {
        if (forceReuploadRequested) {
            logger.info("Full reupload requested.");
            return true;
        }

        if (exportTableExists && exportTableUploadTime.getMonthValue() != newUploadTime.getMonthValue()) {
            logger.info("Starting new month, making full reupload.");
            return true;
        }

        return false;
    }

    /**
     * Нужно ли сделать инкрементальный апдейт.
     */
    boolean needIncrementalUpdate() {
        return exportTableExists && exportTableUploadTime.getDayOfYear() != newUploadTime.getDayOfYear() &&
                newUploadTime.getHour() >= 3;
    }

    /**
     * @return дату и время предыдущего успешного экспорта
     */
    LocalDateTime getExportTableUploadTime() {
        return exportTableUploadTime;
    }

    /**
     * Возвращает дату и время, которые нужно выставить в новой экспортной таблице.
     * Полезно их вычислить до начала экспорта, чтобы можно было сказать, что в таблице есть данные
     * до этой даты.
     */
    String getNewUploadTime() {
        return newUploadTime.toString();
    }

    /**
     * Была ли запрошена принудительная переналивка.
     */
    boolean isForceReuploadRequested() {
        return forceReuploadRequested;
    }
}
