package ru.yandex.direct.jobs.balance.dataimport;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.balance.client.model.response.LinkedClientsItem;
import ru.yandex.direct.core.entity.client.model.ClientBrand;
import ru.yandex.direct.core.entity.client.service.ClientBrandsService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

/**
 * Получение из Биллинга данных о связках клиентов в бренд.
 * Заполняет таблицу {@link ru.yandex.direct.dbschema.ppc.tables.ClientBrands} в PPC,
 * обновляет в ней данные и удаляет данные, которые давно не обновлялись
 */
@Component
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 19),
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 15 */5 * * ?")
@ParametersAreNonnullByDefault
public class ClientBrandsImportJob extends DirectJob {
    /**
     * Время жизни связки клиента с брендом.
     * Если запись не обновлялась в течении указанного периода, то она удаляется.
     */
    static final Duration BRAND_TTL = Duration.ofHours(12);
    private static final float MIN_VALID_BALANCE_RESPONSE_MULTIPLIER = 0.5f;
    /**
     * Ручка умеет отдвать разные типы связи, нас интересуют два:
     * 7 - техническая связка Директа
     * 77 - "бумажный" бренд Директа
     * <p>
     * Подробнее в комментарии https://st.yandex-team.ru/BALANCE-31882#5e58ef5358bb4b2c058da14f
     */
    private static final int DIRECT_LINK = 7;
    private static final int DIRECT_PAPER_BRAND_LINK = 77;

    private static final Logger logger = LoggerFactory.getLogger(ClientBrandsImportJob.class);

    private final BalanceClient balanceClient;
    private final ClientBrandsService clientBrandsService;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientBrandsImportJob(ClientBrandsService clientBrandsService,
                                 BalanceClient balanceClient,
                                 ShardHelper shardHelper) {
        this.clientBrandsService = clientBrandsService;
        this.balanceClient = balanceClient;
        this.shardHelper = shardHelper;
    }

    @Override
    public void execute() {
        int uniqueClientsNum = processBrands();
        int clientsInDatabaseNum = clientBrandsService.getClientBrandsCount();

        if (uniqueClientsNum > clientsInDatabaseNum * MIN_VALID_BALANCE_RESPONSE_MULTIPLIER) {
            deleteOldBrandEntries();
        } else {
            logger.info("Not enough records for delete by TTL");
            setJugglerStatus(JugglerStatus.WARN,
                    "Possible error: Received too few unique clients from Balance::balance_get_direct_brand for " +
                            "delete by TTL");
        }
    }

    private int processBrands() {
        logger.info("Requesting linked clients");

        List<LinkedClientsItem> linkedClientsItems =
                balanceClient.getLinkedClients(List.of(DIRECT_LINK, DIRECT_PAPER_BRAND_LINK));
        logger.info("Got {} linked clients rows", linkedClientsItems.size());

        List<ClientBrand> clientBrandList = convertResponseDataToModel(linkedClientsItems);
        logger.info("Total unique clients in brands is {}", clientBrandList.size());

        List<ClientBrand> validClientBrandList = filterNonexistanceClients(clientBrandList);
        logger.info("Total valid clients in brands is {}", clientBrandList.size());

        logger.info("Inserting data chunk");
        clientBrandsService.replaceClientBrands(validClientBrandList);

        return clientBrandList.size();
    }

    /**
     * Фильтрует список клиентов, оставляя только тех, у которых ClientID корректные и которые известны директу.
     */
    List<ClientBrand> filterNonexistanceClients(List<ClientBrand> clientBrands) {
        List<ClientBrand> validClientBrands = StreamEx.of(clientBrands)
                .filter(c -> c.getClientId() != null)
                .filter(c -> c.getBrandClientId() != null)
                .filter(c -> validId().apply(c.getClientId()) == null)
                .filter(c -> validId().apply(c.getBrandClientId()) == null)
                .toList();

        List<Long> clientIds = StreamEx.of(validClientBrands)
                .flatMap(c -> Stream.of(c.getClientId(), c.getBrandClientId()))
                .distinct()
                .toList();

        Set<Long> unknownClients = EntryStream.of(shardHelper.getShardsByClientIds(clientIds))
                .filterValues(Objects::isNull)
                .keys()
                .toSet();

        logger.info("Invalid/unknown clientIds: " + Arrays.toString(unknownClients.toArray(new Long[0])));

        return StreamEx.of(validClientBrands)
                .remove(brand -> unknownClients.contains(brand.getClientId()))
                .remove(brand -> unknownClients.contains(brand.getBrandClientId()))
                .toList();
    }

    List<ClientBrand> convertResponseDataToModel(Collection<LinkedClientsItem> brandItems) {
        return brandItems.stream().distinct()
                .map(getLinkedClientsItemConverter())
                .collect(Collectors.toList());
    }

    private Function<LinkedClientsItem, ClientBrand> getLinkedClientsItemConverter() {
        LocalDateTime syncTime = LocalDateTime.now();
        return item -> new ClientBrand()
                .withClientId(item.getClientId())
                .withBrandClientId(item.getBrandClientId())
                .withLastSync(syncTime);
    }

    private void deleteOldBrandEntries() {
        LocalDateTime borderDateTime = LocalDateTime.now().minus(BRAND_TTL);
        logger.info("Delete records by TTL (older than {})", borderDateTime);
        int deletedCount = clientBrandsService.deleteEntriesOlderThanDateTime(borderDateTime);

        logger.info("Deleted {} records", deletedCount);
    }
}
