package ru.yandex.direct.jobs.balance.productsimport;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.product.repository.ProductRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.utils.FreshestClusterFinder;

@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_OXID,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 0 2 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ProductsImporter extends DirectJob {
    private final YtProvider ytProvider;
    private final static String productsTablePath = "//home/balance/prod/dict/v_product";
    private static final long CHUNK_SIZE = 1024;
    private static final Logger logger = LoggerFactory.getLogger(ProductsImporter.class);
    private final ProductTableRowConsumer productTableRowConsumer;
    private final ProductRepository productRepository;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public ProductsImporter(YtProvider ytProvider, ProductTableRowConsumer productTableRowConsumer,
                            ProductRepository productRepository, DslContextProvider dslContextProvider) {
        this.ytProvider = ytProvider;
        this.productTableRowConsumer = productTableRowConsumer;
        this.productRepository = productRepository;
        this.dslContextProvider = dslContextProvider;
    }


    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(chooseFreshestCluster(), YtSQLSyntaxVersion.SQLv1);

        for (long chunkIndex = 0; ; chunkIndex++) {
            ytOperator.readTableByRowRange(
                    new YtTable(productsTablePath),
                    productTableRowConsumer,
                    new ProductsTableRow(),
                    CHUNK_SIZE * chunkIndex,
                    CHUNK_SIZE * (chunkIndex + 1));

            var products = productTableRowConsumer.getProducts();
            logger.info("Got {} product records from YT", products.size());

            if (productTableRowConsumer.getReadCount() == 0) {
                break;
            }

            if (products.isEmpty()) {
                continue;
            }

            productRepository.insertNewProducts(dslContextProvider.ppcdict(), products);
            productTableRowConsumer.clear();
        }

    }

    public YtCluster chooseFreshestCluster() {
        return FreshestClusterFinder.getFreshestCluster(ytProvider, new YtTable(productsTablePath),
                List.of(YtCluster.HAHN, YtCluster.ARNOLD));
    }
}
