package ru.yandex.direct.jobs.balanceaggrmigration;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportSpecials;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportSpecialsRepository;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.entity.ppcproperty.model.WalletMigrationStateFlag;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertyNames.BS_CAMPS_ONLY_CAMPS_LIMIT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
class BalanceAggregateMigrationChecker {

    private static final Logger logger = LoggerFactory.getLogger(BalanceAggregateMigrationChecker.class);

    static final int BS_QUEUE_SPECIAL_WALLETS_LIMIT = 30;
    static final Duration MAX_TIME_MIGRATION_DURATION = Duration.ofMinutes(15);
    private static final int DEFAULT_CAMPS_ONLY_MAX_SIZE = 15_000;

    private static final Set<QueueType> ALLOWED_BS_EXPORT_SPECIAL_TYPES =
            ImmutableSet.of(QueueType.HEAVY, QueueType.CAMPS_ONLY, QueueType.BUGGY);

    private final BsExportSpecialsRepository bsExportSpecialsRepository;
    private final BsExportQueueRepository bsExportQueueRepository;
    private final ShardHelper shardHelper;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PpcProperty<Integer> bsCampsOnlyCampsLimitProperty;

    @Autowired
    BalanceAggregateMigrationChecker(
            BsExportSpecialsRepository bsExportSpecialsRepository,
            BsExportQueueRepository bsExportQueueRepository,
            ShardHelper shardHelper, PpcPropertiesSupport ppcPropertiesSupport) {
        this.bsExportSpecialsRepository = bsExportSpecialsRepository;
        this.bsExportQueueRepository = bsExportQueueRepository;
        this.shardHelper = shardHelper;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.bsCampsOnlyCampsLimitProperty = ppcPropertiesSupport.get(BS_CAMPS_ONLY_CAMPS_LIMIT, Duration.ofMinutes(1));
    }


    /**
     * условие запуска джобы.
     * Смотрим на флаг в ppc_properties и последнее время его активации
     */
    boolean isJobEnabled() {
        WalletMigrationStateFlag flag;
        try {
            flag = ppcPropertiesSupport.find(PpcPropertyEnum.WALLET_SUMS_MIGRATION_STATE.getName())
                    .map(prop -> JsonUtils.fromJson(prop, new TypeReference<WalletMigrationStateFlag>() {
                    }))
                    .orElse(WalletMigrationStateFlag.disabled());
        } catch (Exception e) {
            logger.error("Failed to parse json property " + PpcPropertyEnum.WALLET_SUMS_MIGRATION_STATE, e);
            flag = WalletMigrationStateFlag.disabled();
        }

        long currentEpochSeconds = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
        Duration lastMigrationFlagChangedHasPassed = Duration.ofSeconds(currentEpochSeconds - nvl(flag.getTime(), 0L));

        boolean lastMigrationFlagChangedMoreThanMaxDuration =
                lastMigrationFlagChangedHasPassed.compareTo(MAX_TIME_MIGRATION_DURATION) > 0;

        return flag.isEnabled() && lastMigrationFlagChangedMoreThanMaxDuration;
    }


    boolean hasAvailablePlaceInBsExportSpecials(int shard) {
        int walletsInQueue = bsExportSpecialsRepository
                .campaignsByTypeSizeInQueue(shard, QueueType.CAMPS_ONLY, CampaignType.WALLET);
        return BS_QUEUE_SPECIAL_WALLETS_LIMIT > walletsInQueue;
    }

    boolean isWalletAndHasRealCurrency(Campaign campaign) {
        CampaignType type = campaign.getType();
        CurrencyCode currency = campaign.getCurrency();

        return type == CampaignType.WALLET
                && currency.getCurrency().getCode() != CurrencyCode.YND_FIXED;
    }

    boolean checkSumInCampaigns(Campaign wallet, WalletParamsModel walletParams, List<Campaign> campaigns) {
        BigDecimal sums = StreamEx.of(wallet.getSum())
                .append(mapList(campaigns, Campaign::getSum))
                .reduce(BigDecimal::add)
                .orElse(BigDecimal.ZERO);

        return walletParams.getTotalSum().compareTo(sums) == 0;
    }

    boolean allSumsAreZero(List<Campaign> campaigns) {
        return campaigns.stream()
                .allMatch(c -> c.getSum().compareTo(BigDecimal.ZERO) == 0);
    }

    boolean isMigrateStatusCorrect(WalletParamsModel walletParams, BalanceMigrationDirection migrationDirection) {
        AggregatingSumStatus status = walletParams.getAggregateMigrateStatus();
        return migrationDirection == BalanceMigrationDirection.MIGRATE
                ? status == AggregatingSumStatus.NO
                : status == AggregatingSumStatus.YES;
    }

    /**
     * Проверяет, находится ли ОС или кампании под ОС в спец очереди БК (за исключением heavy, buggy и camps_only),
     * и не отправляются ли они в БК прямо сейчас, судя по {@code bs_export_queue.par_id}
     *
     * @param ignoreBsQueue если {@code true}, то не проверяется наличие кампаний в очереди {@code bs_export_queue}
     * @param migrationDirection направление миграции (миграция или откатывание)
     * @return true, если ОС или кампании нельзя мигрировать, т.к. они в специальной очереди,
     * или прямо сейчас отправляются в БК
     */
    boolean isInBsQueues(
            Long walletId,
            List<Campaign> campaigns,
            Map<Long, BigDecimal> chipsCosts,
            boolean ignoreBsQueue,
            BalanceMigrationDirection migrationDirection
    ) {
        // проверяем только ОС и те кампании, у которых есть зачисления в Биллинге.
        // Но не проверяем кампании, у которых кол-во зачислений равно сумме фишек, потраченных до конвертации в валюту.
        // Если sum == campaigns_multicurrency_sums.chips_cost, то в поле SUMCur уже и так отправляется 0,
        // и больше ничего переносить не нужно.
        List<Long> campIdsWithMoney = campaigns.stream()
                .filter(c -> {
                    BigDecimal sumBalance =
                            migrationDirection == BalanceMigrationDirection.MIGRATE ? c.getSum() : c.getSumBalance();

                    if (sumBalance.compareTo(BigDecimal.ZERO) == 0) {
                        return false;
                    }
                    return chipsCosts.getOrDefault(c.getId(), BigDecimal.ZERO).compareTo(sumBalance) != 0;
                })
                .map(Campaign::getId)
                .collect(Collectors.toList());

        List<Long> idsToCheck = StreamEx.of(walletId)
                .append(campIdsWithMoney)
                .toList();

        int shard = shardHelper.getShardByCampaignId(walletId);
        if (!ignoreBsQueue) {
            Set<Long> campaignsIdsInQueue =
                    bsExportQueueRepository.getCampaignIdsInQueueExceptMasterExport(shard, idsToCheck);
            if (!campaignsIdsInQueue.isEmpty()) {
                logger.warn("wallet " + walletId + " has campaigns in bs export queue");
                return true;
            }
        }

        List<BsExportSpecials> bsExportSpecials = bsExportSpecialsRepository.getByCampaignIds(shard, idsToCheck);
        Set<QueueType> wrongQueueTypes = bsExportSpecials.stream()
                .filter(x -> !ALLOWED_BS_EXPORT_SPECIAL_TYPES.contains(x.getType()))
                .map(BsExportSpecials::getType)
                .collect(toSet());
        if (!wrongQueueTypes.isEmpty()) {
            String queueTypes = StringUtils.join(wrongQueueTypes, ", ");
            logger.warn("wallet " + walletId + " has illegal campaign bs queue specials: " + queueTypes);
            return true;
        }

        return false;
    }

    /**
     * Проверяет, что после добавления всех кампаний с деньгами в спец. очередь, она не переполнится.
     * Максимальный размер очереди определяется пропертей
     * {@link ru.yandex.direct.common.db.PpcPropertyNames#BS_CAMPS_ONLY_CAMPS_LIMIT}
     * Если проперти нет, пределом по умолчанию считается {@value DEFAULT_CAMPS_ONLY_MAX_SIZE}
     *
     * @return true, если после добавления кампаний этого ОС специальная очередь не переполнится.
     */
    boolean checkSpecialQueueSize(Long walletId, List<Campaign> campaigns) {
        long campsWithMoneyCount = campaigns.stream()
                .filter(c -> c.getSum().compareTo(BigDecimal.ZERO) != 0
                        || c.getSumBalance().compareTo(BigDecimal.ZERO) != 0)
                .map(Campaign::getId)
                .count();
        int maxSize = bsCampsOnlyCampsLimitProperty.getOrDefault(DEFAULT_CAMPS_ONLY_MAX_SIZE);

        int shard = shardHelper.getShardByCampaignId(walletId);
        int queueSize = bsExportSpecialsRepository.getQueueSize(shard, QueueType.CAMPS_ONLY);
        return (queueSize + campsWithMoneyCount) <= maxSize;
    }
}
