package ru.yandex.direct.jobs.balanceaggrmigration;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.balanceaggrmigration.container.BalanceAggregateMigrationParams;
import ru.yandex.direct.core.entity.balanceaggrmigration.container.BalanceAggregateMigrationResult;
import ru.yandex.direct.core.entity.balanceaggrmigration.lock.AggregateMigrationRedisLockService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignsMulticurrencySumsRepository;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.dbqueue.JobFailedPermanentlyException;
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_DIRECT_BILL_AGG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_0})
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class BalanceAggregateMigrationJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(BalanceAggregateMigrationJob.class);

    private static final Duration REPEAT_AFTER_FAILED_LOCK = Duration.ofMinutes(15);
    private static final Duration REPEAT_AFTER_NOT_VALID_SUMS_OR_BS_QUEUE = Duration.ofMinutes(20);
    private static final int MAX_ATTEMPTS = 30;
    private static final double RELAX_COEFFICIENT = 1;

    private static final int MIGRATION_JOBS_GRAB_ITERATIONS_MAX = 10;

    private final DbQueueService dbQueueService;
    private final AggregateMigrationRedisLockService migrationRedisLockService;
    private final CampaignRepository campaignRepository;
    private final CampaignsMulticurrencySumsRepository campaignsMulticurrencySumsRepository;
    private final ShardHelper shardHelper;
    private final BalanceAggregateMigrationChecker migrationChecker;
    private final BalanceAggregateMigrationService balanceAggregateMigrationService;
    private final WalletParamsRepository walletParamsRepository;
    private final RelaxedWorker relaxedWorker;

    @Autowired
    public BalanceAggregateMigrationJob(
            DbQueueService dbQueueService,
            AggregateMigrationRedisLockService migrationRedisLockService,
            CampaignRepository campaignRepository,
            CampaignsMulticurrencySumsRepository campaignsMulticurrencySumsRepository,
            ShardHelper shardHelper,
            BalanceAggregateMigrationChecker migrationChecker,
            BalanceAggregateMigrationService balanceAggregateMigrationService,
            WalletParamsRepository walletParamsRepository)
    {
        this.dbQueueService = dbQueueService;
        this.migrationRedisLockService = migrationRedisLockService;
        this.campaignRepository = campaignRepository;
        this.campaignsMulticurrencySumsRepository = campaignsMulticurrencySumsRepository;
        this.shardHelper = shardHelper;
        this.migrationChecker = migrationChecker;
        this.balanceAggregateMigrationService = balanceAggregateMigrationService;
        this.walletParamsRepository = walletParamsRepository;
        relaxedWorker = new RelaxedWorker(RELAX_COEFFICIENT);
    }

    @Override
    public void execute() {
        for (int i = 0; i < MIGRATION_JOBS_GRAB_ITERATIONS_MAX; i++) {
            boolean jobFounded = dbQueueJobMigrationIteration();
            if (!jobFounded) {
                break;
            }
        }
    }

    /**
     * Одна итерация выполнения задания из dbQueue со всеми проверками
     *
     * @return нашлась ли задача для обработки
     */
    private boolean dbQueueJobMigrationIteration() {
        if (!migrationChecker.isJobEnabled()) {
            logger.debug("Skip processing. Job is not enabled or not safe time passed after starting.");
            return false;
        }
        if (!migrationChecker.hasAvailablePlaceInBsExportSpecials(getShard())) {
            logger.info("Skip processing. No free place in bs db queue.");
            return false;
        }

        return relaxedWorker.callAndRelax(() -> dbQueueService.grabAndProcessJob(getShard(),
                DbQueueJobTypes.BALANCE_AGGREGATE_MIGRATION,
                BalanceAggregateMigrationChecker.MAX_TIME_MIGRATION_DURATION,
                this::processGrabbedJob,
                MAX_ATTEMPTS,
                this::handleProcessingException)
        );
    }

    /**
     * проверка входных данных и обработка задачи миграции общего счёта
     */
    private BalanceAggregateMigrationResult processGrabbedJob(
            DbQueueJob<BalanceAggregateMigrationParams, BalanceAggregateMigrationResult> jobInfo)
    {
        BalanceAggregateMigrationParams args = jobInfo.getArgs();
        Long jobId = jobInfo.getId();
        Long walletId = args.getWalletId();
        BalanceMigrationDirection migrationDirection =
                args.isRollback() ? BalanceMigrationDirection.ROLLBACK : BalanceMigrationDirection.MIGRATE;
        boolean ignoreBsQueue = args.isIgnoreBsQueue();
        int shard = shardHelper.getShardByClientIdStrictly(jobInfo.getClientId());

        logger.info(
                "start processing job {} on shard {} for wallet {}, direction: {}, ignoreBsQueue: {}, tryCount: {}",
                jobId, shard, walletId, migrationDirection, ignoreBsQueue, jobInfo.getTryCount());

        List<Campaign> walletList = campaignRepository.getCampaigns(shard, singleton(walletId));
        boolean isValid = preValidate(walletId, walletList);
        if (!isValid) {
            String error = "preValidate for job " + jobId + " is failed. Fail job permanently.";
            throw new JobFailedPermanentlyException(error);
        }
        Campaign wallet = walletList.get(0); // проверили, что полученная кампания есть и является общим счётом

        List<WalletParamsModel> walletParamsList =
                walletParamsRepository.get(shard, singletonList(walletId));
        checkState(!walletParamsList.isEmpty(),
                "параметры общего счёта должны быть в базе, кошелёк " + walletId + " существует.");
        WalletParamsModel walletParams = walletParamsList.get(0);

        if (!validateDirection(walletParams, migrationDirection)) {
            String error = "validateDirection for job " + jobId + " failed. Fail job permanently.";
            throw new JobFailedPermanentlyException(error);
        }

        return migrateInLockedSection(shard, wallet, walletParams, jobId, migrationDirection, ignoreBsQueue);
    }

    /**
     * @return true, если общий счет хотят мигрировать в правильную сторону (не мигрируется уже мигрированный ОС)
     */
    private boolean validateDirection(WalletParamsModel walletParams, BalanceMigrationDirection migrationDirection) {
        Long walletId = walletParams.getId();
        if (!migrationChecker.isMigrateStatusCorrect(walletParams, migrationDirection)) {
            logger.error("cannot migrate, wallet has inconsistent migration status and flag of migration direction {}",
                    walletId);
            return false;
        }

        return true;
    }


    private BalanceAggregateMigrationResult handleProcessingException(
            DbQueueJob<BalanceAggregateMigrationParams, BalanceAggregateMigrationResult> jobInfo, String stacktrace)
    {
        logger.error("cannot proccess job {} for wallet {}", jobInfo.getId(), jobInfo.getArgs().getWalletId());
        return BalanceAggregateMigrationResult.error(stacktrace);
    }


    /**
     * секция миграции общего счёта под редис-локом.
     */
    private BalanceAggregateMigrationResult migrateInLockedSection(
            int shard,
            Campaign wallet, WalletParamsModel walletParams,
            Long jobId, BalanceMigrationDirection migrationDirection, boolean ignoreBsQueue
    )
    {
        DistributedLock lock = migrationRedisLockService.lock(wallet.getId());
        if (!lock.isLocked()) {
            logger.warn("cannot get lock for wallet {}, retry job {} later.", wallet.getId(), jobId);
            throw new JobFailedWithTryLaterException(REPEAT_AFTER_FAILED_LOCK);
        }

        BalanceAggregateMigrationResult result = null;
        try {
            List<Long> campaignIds = campaignRepository.getCampaignIdsUnderWallet(shard, wallet.getId());
            List<Campaign> campaigns = campaignRepository.getCampaigns(shard, campaignIds);
            Map<Long, BigDecimal> chipsCosts =
                    campaignsMulticurrencySumsRepository.getCampaignsMulticurrencyChipsCosts(shard, campaignIds);

            boolean isValid = validate(wallet, walletParams, campaigns, chipsCosts, migrationDirection, ignoreBsQueue);
            if (isValid) {
                balanceAggregateMigrationService.migrateWallet(
                        shard, wallet, walletParams, campaigns, chipsCosts, migrationDirection
                );
                result = BalanceAggregateMigrationResult.success();
            }
        } finally {
            migrationRedisLockService.unlock(lock);
        }

        if (result == null) {
            logger.warn("validation is not correct for wallet {} , retry job {} later", wallet.getId(), jobId);
            throw new JobFailedWithTryLaterException(REPEAT_AFTER_NOT_VALID_SUMS_OR_BS_QUEUE);
        }

        return result;
    }

    private boolean preValidate(long walletId, List<Campaign> campaigns) {
        if (campaigns.isEmpty()) {
            logger.warn("campaign {} not found. skip job", walletId);
            return false;
        }

        Campaign campaign = campaigns.get(0);
        if (!migrationChecker.isWalletAndHasRealCurrency(campaign)) {
            logger.warn("skip campaign {}, inappropriate attributes.", walletId);
            return false;
        }

        return true;
    }

    private boolean validate(Campaign wallet, WalletParamsModel walletParams,
            List<Campaign> campaigns,
            Map<Long, BigDecimal> chipsCosts,
            BalanceMigrationDirection migrationDirection, boolean ignoreBsQueue)
    {
        Long walletId = wallet.getId();

        if (!migrationChecker.checkSumInCampaigns(wallet, walletParams, campaigns)) {
            logger.error("cannot migrate, some troubles in sums for walletId {}", walletId);
            return false;
        }
        if (migrationChecker.isInBsQueues(walletId, campaigns, chipsCosts, ignoreBsQueue, migrationDirection)) {
            logger.error("cannot migrate, walletId or campaigns in bs ready for sending {}", walletId);
            return false;
        }
        if (!migrationChecker.checkSpecialQueueSize(walletId, campaigns)) {
            if (!ignoreBsQueue) {
                logger.error("cannot migrate, camps_only special queue size is almost full");
                return false;
            } else {
                logger.warn("camps_only queue overflow");
            }
        }

        // additional checks on rollback
        if (migrationDirection == BalanceMigrationDirection.ROLLBACK) {
            if (!migrationChecker.allSumsAreZero(campaigns)) {
                logger.error("cannot migrate back, campaigns sums are not zero for walletId {}", walletId);
                return false;
            }
        }

        return true;
    }

}
