package ru.yandex.direct.jobs.balanceaggrmigration;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportSpecials;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportSpecialsRepository;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
@ParametersAreNonnullByDefault
class BalanceAggregateMigrationService {

    private final CampaignRepository campaignRepository;
    private final WalletParamsRepository walletParamsRepository;
    private final DslContextProvider dslContextProvider;
    private final BsExportSpecialsRepository bsExportSpecialsRepository;
    private final BsExportQueueRepository bsExportQueueRepository;

    @Autowired
    BalanceAggregateMigrationService(
            CampaignRepository campaignRepository,
            WalletParamsRepository walletParamsRepository,
            DslContextProvider dslContextProvider,
            BsExportSpecialsRepository bsExportSpecialsRepository,
            BsExportQueueRepository bsExportQueueRepository) {
        this.campaignRepository = campaignRepository;
        this.walletParamsRepository = walletParamsRepository;
        this.dslContextProvider = dslContextProvider;
        this.bsExportSpecialsRepository = bsExportSpecialsRepository;
        this.bsExportQueueRepository = bsExportQueueRepository;
    }

    void migrateWallet(int shard, Campaign wallet, WalletParamsModel walletParams,
                       List<Campaign> campaignsUnderWallet,
                       Map<Long, BigDecimal> chipsCosts,
                       BalanceMigrationDirection migrationDirection
    ) {
        if (migrationDirection == BalanceMigrationDirection.ROLLBACK) {
            rollbackWalletInTransaction(shard, wallet, walletParams, campaignsUnderWallet, chipsCosts);
        } else {
            migrateWalletInTransaction(shard, wallet, walletParams, campaignsUnderWallet, chipsCosts);
        }
    }


    // migrate

    void migrateWalletInTransaction(
            int shard, Campaign wallet, WalletParamsModel walletParams,
            List<Campaign> campaignsUnderWallet,
            Map<Long, BigDecimal> chipsCosts
    ) {
        dslContextProvider.ppc(shard).transaction(t -> {
            DSLContext txContext = t.dsl();

            List<AppliedChanges<Campaign>> campaignsAppliedChanges =
                    applyChangesForCampaignsMigrate(campaignsUnderWallet);

            AppliedChanges<Campaign> walletAppliedChanges =
                    applyChangesForWalletMigrate(wallet, walletParams.getTotalSum());

            campaignsAppliedChanges.add(walletAppliedChanges);

            campaignRepository.updateCampaigns(txContext, campaignsAppliedChanges);
            updateWalletParamsOnMigrate(txContext, walletParams);

            addChangedCampaignsToBsSpecialQueue(txContext, campaignsAppliedChanges, chipsCosts,
                    BalanceMigrationDirection.MIGRATE);
        });
    }


    private AppliedChanges<Campaign> applyChangesForWalletMigrate(Campaign wallet, BigDecimal newSumForWallet) {
        ModelChanges<Campaign> changes = new ModelChanges<>(wallet.getId(), Campaign.class);
        changes.process(wallet.getSum(), Campaign.SUM_BALANCE);
        changes.process(newSumForWallet, Campaign.SUM);

        AppliedChanges<Campaign> appliedChanges = changes.applyTo(wallet);
        resetStatusBsSyncedIfSumChanged(appliedChanges);

        return appliedChanges;
    }

    private List<AppliedChanges<Campaign>> applyChangesForCampaignsMigrate(List<Campaign> campaigns) {
        List<AppliedChanges<Campaign>> appliedChangesCampaigns = new ArrayList<>();

        for (Campaign campaign : campaigns) {
            ModelChanges<Campaign> changes = new ModelChanges<>(campaign.getId(), Campaign.class);
            changes.process(campaign.getSum(), Campaign.SUM_BALANCE);
            changes.process(BigDecimal.ZERO, Campaign.SUM);

            AppliedChanges<Campaign> appliedChanges = changes.applyTo(campaign);
            resetStatusBsSyncedIfSumChanged(appliedChanges);

            appliedChangesCampaigns.add(appliedChanges);
        }

        return appliedChangesCampaigns;
    }

    private void updateWalletParamsOnMigrate(DSLContext context, WalletParamsModel walletParams) {
        BigDecimal totalChipsCost = walletParamsRepository.getChipsCostForUpdate(context, walletParams.getId());
        AppliedChanges<WalletParamsModel> walletParamsAppliedChanges =
                appliedChangesForWalletParams(walletParams, totalChipsCost, AggregatingSumStatus.YES);
        walletParamsRepository.update(context, singletonList(walletParamsAppliedChanges));
    }


    // rollback

    void rollbackWalletInTransaction(
            int shard, Campaign wallet, WalletParamsModel walletParams,
            List<Campaign> campaignsUnderWallet,
            Map<Long, BigDecimal> chipsCosts
    ) {
        dslContextProvider.ppc(shard).transaction(t -> {
            DSLContext txContext = t.dsl();

            List<AppliedChanges<Campaign>> campaignsAppliedChanges =
                    applyChangesForCampaignsRollback(campaignsUnderWallet);
            AppliedChanges<Campaign> walletAppliedChanges =
                    applyChangesForWalletRollback(wallet);

            campaignsAppliedChanges.add(walletAppliedChanges);

            campaignRepository.updateCampaigns(txContext, campaignsAppliedChanges);
            updateWalletParamsOnRollback(txContext, walletParams);

            addChangedCampaignsToBsSpecialQueue(txContext, campaignsAppliedChanges, chipsCosts,
                    BalanceMigrationDirection.ROLLBACK);
        });
    }


    private AppliedChanges<Campaign> applyChangesForWalletRollback(Campaign wallet) {
        ModelChanges<Campaign> changes = new ModelChanges<>(wallet.getId(), Campaign.class);
        changes.process(wallet.getSumBalance(), Campaign.SUM);
        changes.process(BigDecimal.ZERO, Campaign.SUM_BALANCE);

        AppliedChanges<Campaign> appliedChanges = changes.applyTo(wallet);
        resetStatusBsSyncedIfSumChanged(appliedChanges);

        return appliedChanges;
    }

    private List<AppliedChanges<Campaign>> applyChangesForCampaignsRollback(List<Campaign> campaigns) {
        List<AppliedChanges<Campaign>> appliedChangesCampaigns = new ArrayList<>();

        for (Campaign campaign : campaigns) {
            ModelChanges<Campaign> changes = new ModelChanges<>(campaign.getId(), Campaign.class);
            changes.process(campaign.getSumBalance(), Campaign.SUM);
            changes.process(BigDecimal.ZERO, Campaign.SUM_BALANCE);

            AppliedChanges<Campaign> appliedChanges = changes.applyTo(campaign);
            resetStatusBsSyncedIfSumChanged(appliedChanges);

            appliedChangesCampaigns.add(appliedChanges);
        }

        return appliedChangesCampaigns;
    }

    private void updateWalletParamsOnRollback(DSLContext context, WalletParamsModel walletParams) {
        AppliedChanges<WalletParamsModel> walletParamsAppliedChanges =
                appliedChangesForWalletParams(walletParams, BigDecimal.ZERO, AggregatingSumStatus.NO);
        walletParamsRepository.update(context, singletonList(walletParamsAppliedChanges));
    }

    // general

    /**
     * Вычисляет, у каких кампаний поменялась сумма зачислений, и кладет их в спец очередь
     * {@code camps_only}, а у тех кампаний, которые уже лежат в {@code bs_export_queue}, сбрасывает
     * значение {@code queue_time}.
     * <p>
     * Спец очередь нужна для того, чтобы суметь в одной пачке отправить ОС и все его кампании в БК.
     * Это достигается тем, что воркер {@code bsClientData}, обрабатывающий эту спец. очередь,
     * отправляет только кампании, без рекламных материалов.
     * <p>
     * Сбрасывать время {@code bs_export_queue.queue_time} оказывается полезно для мониторинга -
     * так появляется возожность измерить, сколько времени кампания лежит в спец. очереди.
     * В целом это довольно наглый ход, но считаем его здесь допустимым, т.к. мы мониторим размер
     * спец. очереди, и кладем кампанию в нее всего один раз в жизни.
     *
     * @param txContext               открытая транзакция БД
     * @param campaignsAppliedChanges список изменений кампаний и ОС.
     */
    private void addChangedCampaignsToBsSpecialQueue(
            DSLContext txContext,
            List<AppliedChanges<Campaign>> campaignsAppliedChanges,
            Map<Long, BigDecimal> chipsCosts,
            BalanceMigrationDirection migrationDirection
    ) {
        List<Long> campaignIdsToBs = campaignsAppliedChanges.stream()
                .filter(ac -> {
                    // не нужно ставить кампанию на срочную переотправку, если сумма зачислений не менялась
                    if (!ac.changed(Campaign.SUM)) {
                        return false;
                    }
                    // также не нужно, если сумма зачислений равна стоимости фишек до конвертации, т.к. при этом
                    // в SUMCur и так едет 0 в транспорте в БК
                    BigDecimal sumBalance =
                            migrationDirection == BalanceMigrationDirection.MIGRATE ?
                                    ac.getModel().getSumBalance() :
                                    ac.getModel().getSum();
                    return chipsCosts.getOrDefault(ac.getModel().getId(), BigDecimal.ZERO).compareTo(sumBalance) != 0;
                })
                .map(ac -> ac.getModel().getId())
                .collect(toList());

        Function<Long, BsExportSpecials> campaignIdToBsExportSpecialConverter = id ->
                new BsExportSpecials().withCampaignId(id).withType(QueueType.CAMPS_ONLY);

        List<BsExportSpecials> bsExportSpecials = mapList(campaignIdsToBs, campaignIdToBsExportSpecialConverter);
        checkNotNull(bsExportSpecials);

        bsExportSpecialsRepository.add(txContext, bsExportSpecials);
        bsExportQueueRepository.resetQueueTime(txContext, campaignIdsToBs);
    }

    private AppliedChanges<WalletParamsModel> appliedChangesForWalletParams(WalletParamsModel walletParams,
                                                                            BigDecimal totalChipsCost,
                                                                            AggregatingSumStatus status) {
        ModelChanges<WalletParamsModel> changes = new ModelChanges<>(walletParams.getId(), WalletParamsModel.class);
        changes.process(totalChipsCost, WalletParamsModel.TOTAL_CHIPS_COST);
        changes.process(status, WalletParamsModel.AGGREGATE_MIGRATE_STATUS);

        return changes.applyTo(walletParams);
    }

    private void resetStatusBsSyncedIfSumChanged(AppliedChanges<Campaign> appliedChanges) {
        if (appliedChanges.changed(Campaign.SUM)) {
            appliedChanges.modify(Campaign.STATUS_BS_SYNCED, StatusBsSynced.NO);
        }
    }
}
