package ru.yandex.direct.jobs.balanceaggrmigration.monitoring;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.AggregateFunction;
import org.jooq.Field;
import org.jooq.Record3;
import org.jooq.Result;
import org.jooq.SelectHavingStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.DirectGraphiteUtils;
import ru.yandex.direct.core.entity.product.model.ProductSimple;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.core.entity.product.repository.ProductsCache;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.graphite.GraphiteClient;
import ru.yandex.direct.graphite.GraphiteMetricsBuffer;
import ru.yandex.direct.jobs.balanceaggrmigration.monitoring.model.ClientStat;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.DateTimeUtils;

import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.WalletCampaigns.WALLET_CAMPAIGNS;

@ParametersAreNonnullByDefault
//@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 180),
//        needCheck = ProductionOnly.class, tags = {GROUP_INTERNAL_SYSTEMS})
// @Hourglass(periodInSeconds = 3600, needSchedule = NonProductionEnvironment.class)
// todo andreymak переделать на YQL + Solomon и включить, когда начнём переводить всех клиентов на объединённую схему
public class BalanceAggregateMigrationStatisticJob extends DirectJob {

    // db fields&criteria

    private static final Field<Long> CLIENT_COUNTER_FIELD = DSL.countDistinct(CAMPAIGNS.CLIENT_ID).cast(Long.class);

    private static final Field<Long> CLIENT_BILLING_AGGREGATE_COUNTER_FIELD = DSL.countDistinct(
            DSL.iif(CAMPAIGNS.TYPE.eq(CampaignsType.billing_aggregate), CAMPAIGNS.CLIENT_ID, DSL.field("null"))
    ).cast(Long.class);

    private static final Set<Long> TEXT_RUB_PRODUCT_IDS = ImmutableSet.of(1475L, 503162L);

    private static final Set<ProductType> PRODUCTS_TYPES =
            ImmutableSet.of(ProductType.TEXT, ProductType.CPM_BANNER, ProductType.CPM_DEALS, ProductType.CPM_VIDEO);


    // graphite configs

    private static final String GRAPHITE_PATH = "jobs.migrating.balanceAggregate.stat";

    private static final String WALLET_STAT_NAME = "wallet";
    private static final String BILLING_AGGREGATE_STAT_NAME = "billingAggregateClients";
    private static final String TOTAL_CAMPAIGNS_STAT_NAME = "totalCampaignsClients";

    private static final String TOTAL_WALLETS_PROP = "Total";


    // components

    private final GraphiteClient graphiteClient;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final ProductsCache productsCache;


    @Autowired
    public BalanceAggregateMigrationStatisticJob(GraphiteClient graphiteClient,
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            ProductsCache productsCache)
    {
        this.graphiteClient = graphiteClient;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.productsCache = productsCache;
    }

    @Override
    public void execute() {
        GraphiteMetricsBuffer metrics = DirectGraphiteUtils.defaultBuffer().with(GRAPHITE_PATH);

        long timestamp = DateTimeUtils.getNowEpochSeconds();

        Map<String, Integer> walletsStat = processWalletStat();
        walletsStat.forEach((name, value) ->
                metrics.with(WALLET_STAT_NAME).add(name, value, timestamp));

        Map<ProductType, ClientStat> billingAggregateStat = processBillingAggregateStat();
        billingAggregateStat.forEach((type, stat) -> {
            metrics.with(TOTAL_CAMPAIGNS_STAT_NAME).add(type.name(), stat.getTotalClients(), timestamp);
            metrics.with(BILLING_AGGREGATE_STAT_NAME).add(type.name(), stat.getBillingAggregateClients(), timestamp);
        });

        graphiteClient.send(metrics);
    }

    /**
     * Статистика количества кошельков по статусам миграций и общее кол-во
     * (без учёта фишковых счетов)
     */
    private Map<String, Integer> processWalletStat() {
        Map<String, Integer> result = new HashMap<>();
        shardHelper.forEachShard(shard -> {
            Map<String, Integer> walletsCounterByStatusMap = getWalletStatByMigrationStatus(shard);
            walletsCounterByStatusMap.forEach((status, counter) ->
                    result.merge(status, counter, (prev, actual) -> prev + actual));
        });

        // добавить статистику по общему количеству ОС во всех статусах в результат
        Integer total = result.values().stream().mapToInt(Integer::intValue).sum();
        result.put(TOTAL_WALLETS_PROP, total);

        return result;
    }

    /**
     * Получить статистику количества общих счетов на заданном шарде с разбивкой по статусу миграции.
     */
    private Map<String, Integer> getWalletStatByMigrationStatus(int shard) {
        AggregateFunction<Integer> counterField = DSL.count(WALLET_CAMPAIGNS.WALLET_CID);

        return dslContextProvider.ppc(shard)
                .select(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED, counterField)
                .from(WALLET_CAMPAIGNS)
                .join(CAMPAIGNS).on(WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CURRENCY.ne(CampaignsCurrency.YND_FIXED))
                .groupBy(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED)
                .fetchMap(r -> r.get(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED).name(), r -> r.get(counterField));
    }

    /**
     * Статистика биллинговых агрегатов по типам
     */
    private Map<ProductType, ClientStat> processBillingAggregateStat() {
        Map<Long, ProductType> filteredProductIdsToType = StreamEx.of(productsCache.getSimpleProducts())
                .filter(p -> PRODUCTS_TYPES.contains(p.getType()))
                .toMap(ProductSimple::getId, ProductSimple::getType);

        Map<Long, ClientStat> clientsStat = new HashMap<>();

        shardHelper.forEachShard(shard -> {
            SelectHavingStep<Record3<Long, Long, Long>> stepTextRubProduct = dslContextProvider.ppc(shard)
                    .select(DSL.max(CAMPAIGNS.PRODUCT_ID).as(CAMPAIGNS.PRODUCT_ID),
                            CLIENT_COUNTER_FIELD,
                            CLIENT_BILLING_AGGREGATE_COUNTER_FIELD
                    )
                    .from(CAMPAIGNS)
                    .where(CAMPAIGNS.PRODUCT_ID.in(TEXT_RUB_PRODUCT_IDS))
                    .and(CAMPAIGNS.CURRENCY.ne(CampaignsCurrency.YND_FIXED));

            SelectHavingStep<Record3<Long, Long, Long>> stepOtherProducts = dslContextProvider.ppc(shard)
                    .select(CAMPAIGNS.PRODUCT_ID,
                            CLIENT_COUNTER_FIELD,
                            CLIENT_BILLING_AGGREGATE_COUNTER_FIELD
                    )
                    .from(CAMPAIGNS)
                    .where(CAMPAIGNS.PRODUCT_ID.isNotNull()
                            .and(CAMPAIGNS.PRODUCT_ID.in(filteredProductIdsToType.keySet()))
                            .and(CAMPAIGNS.PRODUCT_ID.notIn(TEXT_RUB_PRODUCT_IDS)))
                    //.and(CAMPAIGNS.TYPE.in(CAMPAIGN_TYPES))
                    .and(CAMPAIGNS.CURRENCY.ne(CampaignsCurrency.YND_FIXED))
                    .groupBy(CAMPAIGNS.PRODUCT_ID);

            Result<Record3<Long, Long, Long>> recordResult = stepTextRubProduct.union(stepOtherProducts).fetch();

            recordResult.forEach(r -> clientsStat.merge(
                    r.get(CAMPAIGNS.PRODUCT_ID),
                    new ClientStat(r.get(CLIENT_COUNTER_FIELD), r.get(CLIENT_BILLING_AGGREGATE_COUNTER_FIELD)),
                    ClientStat::add));
        });


        // merge by product type
        return EntryStream.of(clientsStat)
                .mapKeys(filteredProductIdsToType::get)
                .toMap(ClientStat::add);
    }
}

