package ru.yandex.direct.jobs.banner;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.domain.model.BannerUpdateAggeregatorDomainParams;
import ru.yandex.direct.core.entity.domain.model.BannerUpdateAggeregatorDomainResult;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.banner.service.BannerUpdateAggeregatorDomainService;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.ThreadUtils;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.UPDATE_AGGREGATOR_DOMAINS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_KUHTICH,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_2})
@Hourglass(periodInSeconds = 60)
@ParametersAreNonnullByDefault
public class BannerUpdateAggeregatorDomainJob extends DirectShardedJob {
    private static final int JOB_RETRIES = 2;
    private static final int RETRY_TIMEOUT = 10;
    private static final Duration GRABBED_TIME = Duration.ofHours(2);

    private static final Logger logger = LoggerFactory.getLogger(BannerUpdateAggeregatorDomainJob.class);

    private final DbQueueService dbQueueService;
    private final BannerUpdateAggeregatorDomainService bannerUpdateAggeregatorDomainService;

    @Autowired
    public BannerUpdateAggeregatorDomainJob(DbQueueService dbQueueService,
            BannerUpdateAggeregatorDomainService bannerUpdateAggeregatorDomainService)
    {
        this.dbQueueService = dbQueueService;
        this.bannerUpdateAggeregatorDomainService = bannerUpdateAggeregatorDomainService;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    BannerUpdateAggeregatorDomainJob(int shard, DbQueueService dbQueueService,
            BannerUpdateAggeregatorDomainService bannerUpdateAggeregatorDomainService)
    {
        super(shard);
        this.dbQueueService = dbQueueService;
        this.bannerUpdateAggeregatorDomainService = bannerUpdateAggeregatorDomainService;
    }

    @Override
    public void execute() {
        while (dbQueueService.grabAndProcessJob(getShard(), UPDATE_AGGREGATOR_DOMAINS, GRABBED_TIME,
                this::processGrabbedJob, JOB_RETRIES, this::onError)) {
            ThreadUtils.sleep(Duration.ofSeconds(RETRY_TIMEOUT).toMillis());
        }
    }

    private BannerUpdateAggeregatorDomainResult processGrabbedJob(DbQueueJob<BannerUpdateAggeregatorDomainParams,
            BannerUpdateAggeregatorDomainResult> jobInfo)
    {
        logger.info("Start processing jobId={}, attempt={}", jobInfo.getId(), jobInfo.getTryCount());
        bannerUpdateAggeregatorDomainService.updateAggregatorDomains(getShard(), jobInfo.getArgs().getDomain());
        return new BannerUpdateAggeregatorDomainResult();
    }

    private BannerUpdateAggeregatorDomainResult onError(
            DbQueueJob<BannerUpdateAggeregatorDomainParams, BannerUpdateAggeregatorDomainResult> jobInfo,
            String stacktrace)
    {
        return new BannerUpdateAggeregatorDomainResult().withError(stacktrace);
    }
}
