package ru.yandex.direct.jobs.banner;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static java.time.LocalDateTime.now;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper.YANDEX_DOMAINS;
import static ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper.YANDEX_URLS_PATTERN;
import static ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper.YANDEX_URLS_REGEX;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Ищем в базе баннеры со ссылками на сервисы Яндекса, для которых неверно посчитан домен. Такое бывает в двух случаях:
 * <p><ul>
 * <li>ссылка ведет на сервис, для которого должен быть посчитан специальный сервисный домен
 * (cм. {@link PpcPropertyNames#YANDEX_URL_TO_SERVICE_MAPPINGS}), но по каким-то причинам домен другой
 * ({@code yandex.ru} или типа того). Для таких баннеров нужно пересчтать домены;</li>
 * <li>ссылка ведет на сервис Яндекса, но этот сервис отсутсвует и в списке сервисов, для которых есть специальный
 * сервисный домен {@link PpcPropertyNames#YANDEX_URL_TO_SERVICE_MAPPINGS}, и в списке сервисов, для которых не нужен
 * специальный сервисный домен{@link PpcPropertyNames#YANDEX_SERVICES_WITHOUT_SPECIAL_DOMAIN}. Ожидается, что это
 * какой-то новый сервис, про который мы раньше ничего не знали, и его нужно поместить в один из вышеупомянутых списков.
 * </li>
 * </ul>
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(days = 2),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.LOGIN_SIVKOV, NotificationRecipient.LOGIN_KUHTICH},
                method = NotificationMethod.EMAIL
        ),
        tags = {DIRECT_PRIORITY_2})
@Hourglass(cronExpression = "0 0 23 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class InvalidYandexServicesDomainsMonitoringJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(InvalidYandexServicesDomainsMonitoringJob.class);

    // заменяем non-capturing groups на обычные, чтобы можно было скормить в MySQL
    private static final String YANDEX_URLS_REGEX_SQL = YANDEX_URLS_REGEX.replaceAll("\\(\\?:", "(");

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PpcProperty<Set<String>> servicesWithoutSpecialDomain;
    private final PpcProperty<Map<String, String>> yandexUrlToServiceMappings;

    public InvalidYandexServicesDomainsMonitoringJob(ShardHelper shardHelper, DslContextProvider dslContextProvider,
                                                     PpcPropertiesSupport ppcPropertiesSupport) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        servicesWithoutSpecialDomain = ppcPropertiesSupport.get(PpcPropertyNames.YANDEX_SERVICES_WITHOUT_SPECIAL_DOMAIN,
                Duration.ofMinutes(10));
        yandexUrlToServiceMappings = ppcPropertiesSupport.get(PpcPropertyNames.YANDEX_URL_TO_SERVICE_MAPPINGS,
                Duration.ofMinutes(10));
    }

    @Override
    public void execute() {
        PpcProperty<LocalDateTime> lastRunProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.INVALID_YANDEX_SERVICES_DOMAINS_MONITORING_LAST_RUN_TIME);
        LocalDateTime currentTime = now();
        LocalDateTime lastRunTime = lastRunProperty.getOrDefault(currentTime.minusDays(1));

        Map<String, List<Long>> invalidDomainsInfo = fetchInvalidDomainsInfo(lastRunTime);
        List<String> unknownServices = filterList(invalidDomainsInfo.keySet(), this::isServiceUnknown);
        Map<String, List<Long>> bannersWithImproperDomains = EntryStream.of(invalidDomainsInfo)
                .filterKeys(this::isServiceWithSpecialDomain)
                .toMap();

        sendNotification(bannersWithImproperDomains, unknownServices);

        lastRunProperty.set(currentTime);
    }

    private boolean isServiceUnknown(String service) {
        return !isServiceWithSpecialDomain(service) &&
                !servicesWithoutSpecialDomain.getOrDefault(emptySet()).contains(service);
    }

    private boolean isServiceWithSpecialDomain(String service) {
        return yandexUrlToServiceMappings.getOrDefault(BannersUrlHelper.YANDEX_URL_TO_SERVICE_MAPPINGS)
                .containsKey(service);
    }

    private Map<String, List<Long>> fetchInvalidDomainsInfo(LocalDateTime startTime) {
        Map<Integer, Map<String, List<Long>>> invalidDomainsInfoByShard = shardHelper.forEachShardSequential(shard ->
            dslContextProvider.ppc(shard)
                    .select(BANNERS.BID, BANNERS.HREF)
                    .from(BANNERS)
                    .where(
                            BANNERS.REVERSE_DOMAIN.in(getReversedYandexDomains()),
                            BANNERS.LAST_CHANGE.ge(startTime),
                            BANNERS.HREF.likeRegex(YANDEX_URLS_REGEX_SQL)
                    )
                    .fetchGroups(
                            r -> extractYandexService(r.get(BANNERS.HREF)),
                            r -> r.get(BANNERS.BID)
                    )
        );

        return StreamEx.of(invalidDomainsInfoByShard.values())
                .flatMap(EntryStream::of)
                .filter(entry -> entry.getKey() != null)
                .toMap(Map.Entry::getKey, Map.Entry::getValue,
                        (left, right) -> Stream.concat(left.stream(), right.stream()).collect(toList()));
    }

    private List<String> getReversedYandexDomains() {
        return StreamEx.of(YANDEX_DOMAINS)
                .map(StringUtils::reverse)
                .toFlatList(reverseDomain -> List.of(reverseDomain, reverseDomain + ".www", reverseDomain + ".m"));
    }

    @Nullable
    private String extractYandexService(String href) {
        var m = YANDEX_URLS_PATTERN.matcher(href);
        return m.matches() ? m.group(2) : null;
    }

    private void sendNotification(Map<String, List<Long>> bannersWithImproperDomains, List<String> unknownServices) {
        if (unknownServices.isEmpty() && bannersWithImproperDomains.isEmpty()) {
            setJugglerStatus(JugglerStatus.OK, "Все в порядке.");
            return;
        }

        String message = "";
        if (!unknownServices.isEmpty()) {
            logger.info("Unknown services: {}", unknownServices);
            message += "Обнаружены незарегистрированные сервисы Яндекса:" + unknownServices + "\\n\\n";
        }
        if (!bannersWithImproperDomains.isEmpty()) {
            logger.info("Banners with improper domains: {}", bannersWithImproperDomains);
            message += "Обнаружены баннеры с неправильным доменом:\\n\\n" +
                    EntryStream.of(bannersWithImproperDomains)
                            .mapKeyValue((service, bids) -> "\\tСервис: " + service + "\\n\\tБаннеры: " + bids)
                            .joining("\n\n");
        }
        setJugglerStatus(JugglerStatus.CRIT, message);
    }
}
