package ru.yandex.direct.jobs.banner;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.ToIntFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.image.repository.BannerImagesUploadsRepository;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassStretchPeriod;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Удаляем старые временные изображения из PPC.banner_images_uploads и удалённые не отправленные в БК картинки из PPC
 * .banner_images. Кроме того удаляем неиспользуемые форматы картинок из PPC.banner_images_formats.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 8),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
// Запускается ежедневно в 4:15 (смещена от ppcRemoveBannerImagesPPC.pl)
@Hourglass(cronExpression = "0 15 4 * * ?", needSchedule = TypicalEnvironment.class)
@HourglassStretchPeriod()
@ParametersAreNonnullByDefault
class RemoveBannerImagesJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(RemoveBannerImagesJob.class);

    static final Duration UPLOADS_MAX_AGE_HOURS = Duration.ofHours(48);
    static final Duration IMAGES_MAX_AGE_HOURS = Duration.ofHours(48);
    private static final int BANNER_IMAGES_CHUNK = 1_000;
    private static final int BANNER_IMAGES_FORMATS_LIMIT = 100_000;
    private final Duration msBeforeDeleteBannerImageFormat;
    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(3.0);

    private final BannerImageRepository bannerImageRepository;
    private final MdsFileService mdsFileService;
    private final BannerImageFormatRepository bannerImageFormatRepository;
    private final BannerImagesUploadsRepository bannerImagesUploadsRepository;

    @Autowired
    public RemoveBannerImagesJob(
            BannerImageRepository bannerImageRepository, MdsFileService mdsFileService,
            BannerImageFormatRepository bannerImageFormatRepository,
            BannerImagesUploadsRepository bannerImagesUploadsRepository) {
        this.bannerImageRepository = bannerImageRepository;
        this.mdsFileService = mdsFileService;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.bannerImagesUploadsRepository = bannerImagesUploadsRepository;
        msBeforeDeleteBannerImageFormat = Duration.ofMinutes(5);
    }

    RemoveBannerImagesJob(int shard,
                          BannerImageRepository bannerImageRepository, MdsFileService mdsFileService,
                          BannerImageFormatRepository bannerImageFormatRepository,
                          BannerImagesUploadsRepository bannerImagesUploadsRepository,
                          Duration msBeforeDeleteBannerImageFormat) {
        super(shard);
        this.bannerImageRepository = bannerImageRepository;
        this.mdsFileService = mdsFileService;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.bannerImagesUploadsRepository = bannerImagesUploadsRepository;
        this.msBeforeDeleteBannerImageFormat = msBeforeDeleteBannerImageFormat;
    }

    @Override
    public void execute() {
        cleanData(LocalDateTime.now().minus(UPLOADS_MAX_AGE_HOURS),
                borderDate -> bannerImagesUploadsRepository.getIdsByDateAdded(getShard(), borderDate),
                ids -> bannerImagesUploadsRepository.deleteByIds(getShard(), ids), "banner_images_uploads");

        cleanData(LocalDateTime.now().minus(IMAGES_MAX_AGE_HOURS),
                borderDate -> bannerImageRepository.getBannerImagesIdsForDelete(getShard(), borderDate),
                ids -> bannerImageRepository.deleteBannerImagesByIds(getShard(), ids), "banner_images");

        cleanBannerImagesFormats();

        mdsFileService.deleteOldMdsFiles(getShard(), LocalDateTime.now().minus(UPLOADS_MAX_AGE_HOURS),
                MdsStorageType.BANNER_IMAGES_UPLOADS);
    }

    private void cleanData(LocalDateTime borderDate, Function<LocalDateTime, List<Long>> collectData,
                           ToIntFunction<List<Long>> deleteData, String tableName) {
        List<Long> ids = collectData.apply(borderDate);
        logger.info("Cleaning {} records from {} table with borderDate {}", ids.size(), tableName, borderDate);
        final AtomicInteger countDeleted = new AtomicInteger();
        for (List<Long> chunk : Iterables.partition(ids, BANNER_IMAGES_CHUNK)) {
            relaxedWorker.runAndRelax(() -> {
                countDeleted.addAndGet(deleteData.applyAsInt(chunk));
                logger.debug("deleted {} records from  {} table", chunk, tableName);
            });
        }
        logger.info("Deleted {} items from {} table", countDeleted, tableName);
    }

    void cleanBannerImagesFormats() {
        List<String> bannerImagesFormats = bannerImageFormatRepository.getHashesForDelete(getShard(),
                BANNER_IMAGES_FORMATS_LIMIT);
        logger.info("Cleaning {} banner_images_formats records", bannerImagesFormats.size());
        if (bannerImagesFormats.isEmpty()) {
            return;
        }

        // картинка и её формат не всегда создаются в одной транзакции и в правильном порядке
        // например, при копировании картинки сначала создаётся формат, а потом картинка
        // чтобы не потерять свежедобавленный формат, даём время его привязть к картинке
        logger.info("Sleeping for {} minutes", msBeforeDeleteBannerImageFormat.toMinutes());
        try {
            Thread.sleep(msBeforeDeleteBannerImageFormat.toMillis());
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }

        final AtomicInteger countDeleted = new AtomicInteger();
        for (List<String> chunk : Iterables.partition(bannerImagesFormats, BANNER_IMAGES_CHUNK)) {
            relaxedWorker.runAndRelax(() -> {
                countDeleted.addAndGet(bannerImageFormatRepository.deleteByHashesWithoutRelations(getShard(), chunk));
                logger.debug("deleted {} records from  banner_images_formats table", chunk);
            });
        }
        logger.info("Deleted {} items from banner_images_formats table", countDeleted);
    }
}
