package ru.yandex.direct.jobs.banner.language;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.BannerWithLanguage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.Language;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.banner.type.language.BannerLanguageConverter;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.function.Function.identity;

abstract class AbstractFillBannersLanguageJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(AbstractFillBannersLanguageJob.class);

    static final int BANNERS_TO_PROCESS_PER_ITERATION = 2000;

    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    protected final BannerCommonRepository bannerCommonRepository;
    private final BannerTextExtractor bannerTextExtractor;
    private final QueryrecService queryrecService;
    private final DslContextProvider dslContextProvider;
    private final ClientRepository clientRepository;

    AbstractFillBannersLanguageJob(BannerTypedRepository bannerTypedRepository,
                                   BannerRelationsRepository bannerRelationsRepository,
                                   BannerCommonRepository bannerCommonRepository,
                                   BannerTextExtractor bannerTextExtractor,
                                   QueryrecService queryrecService,
                                   DslContextProvider dslContextProvider, ClientRepository clientRepository) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerTextExtractor = bannerTextExtractor;
        this.queryrecService = queryrecService;
        this.dslContextProvider = dslContextProvider;
        this.clientRepository = clientRepository;
    }

    /**
     * Конструктор для тестов.
     */
    AbstractFillBannersLanguageJob(int shard, BannerTypedRepository bannerTypedRepository,
                                   BannerRelationsRepository bannerRelationsRepository,
                                   BannerCommonRepository bannerCommonRepository,
                                   BannerTextExtractor bannerTextExtractor,
                                   QueryrecService queryrecService,
                                   DslContextProvider dslContextProvider, ClientRepository clientRepository) {
        super(shard);

        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerTextExtractor = bannerTextExtractor;
        this.queryrecService = queryrecService;
        this.dslContextProvider = dslContextProvider;
        this.clientRepository = clientRepository;
    }

    /**
     * Заполняет язык для заданных баннеров.
     *
     * @param shard шард
     * @param bids  баннеры, которым нужно заполнить язык
     */
    void processBanners(int shard, Collection<Long> bids) {
        Map<Long, BannerWithSystemFields> bannerWithEmptyLanguageByBid =
                StreamEx.of(bannerTypedRepository.getStrictlyFullyFilled(shard, bids, BannerWithSystemFields.class))
                        .filter(banner -> isEmptyLanguage(banner.getLanguage()))
                        .toMap(BannerWithSystemFields::getId, identity());

        if (bannerWithEmptyLanguageByBid.isEmpty()) {
            return;
        }

        Map<Long, ClientId> clientIdByBannerId =
                EntryStream.of(bannerRelationsRepository.getClientIdsByBannerIds(shard, bids))
                        .nonNullValues()
                        .toMap();

        Map<ClientId, Long> clientRegionIdByClientId =
                EntryStream.of(clientRepository.getCountryRegionIdByClientId(shard, clientIdByBannerId.values()))
                        .nonNullValues()
                        .mapKeys(ClientId::fromLong)
                        .toMap();

        var bannerToExtractedText = bannerTextExtractor.extractTexts(bannerWithEmptyLanguageByBid.values());

        Map<Long, Language> recognizedNonEmptyLanguageByBid = EntryStream.of(bannerWithEmptyLanguageByBid)
                .mapValues(bannerToExtractedText::get)
                .mapToValue((bannerId, text) -> {
                    ClientId clientId = clientIdByBannerId.get(bannerId);

                    if (clientId == null) {
                        logger.error("No client for banner {}", bannerId);
                    }

                    Long clientRegionId = clientRegionIdByClientId.get(clientId);

                    if (clientRegionId == null) {
                        logger.warn("No region for client {}", clientId);
                    }

                    return queryrecService.recognize(text, clientId, clientRegionId);
                })
                .mapValues(BannerLanguageConverter::convertLanguage)
                .filterValues(language -> !isEmptyLanguage(language))
                .toMap();

        dslContextProvider.ppcTransaction(shard, configuration -> {
            DSLContext dslContext = DSL.using(configuration);
            Collection<Long> lockedBids = bannerCommonRepository.lockBannersForUpdate(dslContext,
                    recognizedNonEmptyLanguageByBid.keySet());
            var lockedUnchangedBannersToFillLanguage =
                    StreamEx.of(bannerTypedRepository.getStrictlyFullyFilled(dslContext, lockedBids,
                            BannerWithSystemFields.class))
                            .filter(lockedBanner -> lockedBanner
                                    .equals(bannerWithEmptyLanguageByBid.get(lockedBanner.getId())))
                            .filter(lockedUnchangedBanner ->
                                    !isEmptyLanguage(recognizedNonEmptyLanguageByBid
                                            .get(lockedUnchangedBanner.getId())))
                            .toList();

            List<AppliedChanges<BannerWithLanguage>> appliedChanges =
                    StreamEx.of(lockedUnchangedBannersToFillLanguage)
                            .select(BannerWithLanguage.class)
                            .map(banner -> new ModelChanges<>(banner.getId(), BannerWithLanguage.class)
                                    .process(recognizedNonEmptyLanguageByBid.get(banner.getId()),
                                            BannerWithLanguage.LANGUAGE)
                                    .applyTo(banner))
                            .toList();
            bannerCommonRepository.updateBannersLanguages(dslContext, appliedChanges);
        });
    }

    private static boolean isEmptyLanguage(@Nullable Language language) {
        return language == null || language == Language.YES || language == Language.NO || language == Language.UNKNOWN;
    }
}
