package ru.yandex.direct.jobs.banner.language;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.scheduler.Hourglass;

import static ru.yandex.direct.common.db.PpcPropertyNames.FILL_BANNERS_LANGUAGE_JOB_ENABLED;
import static ru.yandex.direct.common.db.PpcPropertyNames.FILL_BANNERS_LANGUAGE_JOB_SLEEP_COEFFICIENT;
import static ru.yandex.direct.common.db.PpcPropertyNames.minBidWithProcessedLanguage;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоба для заполнения языка баннеров.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 60 * 2, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class FillBannersLanguageJob extends AbstractFillBannersLanguageJob {

    private Logger logger = LoggerFactory.getLogger(FillBannersLanguageJob.class);

    // Работает примерно 17 минут (не учитывая время на Thread.sleep())
    private static final int ITERATIONS_PER_EXECUTE = 250;

    private final PpcPropertiesSupport ppcPropertiesSupport;

    private final PpcProperty<Boolean> jobEnabledProperty;
    private final PpcProperty<Double> sleepCoefficientProperty;

    @Autowired
    public FillBannersLanguageJob(BannerTypedRepository bannerTypedRepository,
                                  BannerRelationsRepository bannerRelationsRepository,
                                  BannerCommonRepository bannerCommonRepository,
                                  BannerTextExtractor bannerTextExtractor,
                                  PpcPropertiesSupport ppcPropertiesSupport,
                                  QueryrecService queryrecService, DslContextProvider dslContextProvider,
                                  ClientRepository clientRepository) {
        super(bannerTypedRepository, bannerRelationsRepository, bannerCommonRepository, bannerTextExtractor,
                queryrecService, dslContextProvider, clientRepository);

        this.ppcPropertiesSupport = ppcPropertiesSupport;

        jobEnabledProperty = ppcPropertiesSupport.get(FILL_BANNERS_LANGUAGE_JOB_ENABLED);
        sleepCoefficientProperty = ppcPropertiesSupport.get(FILL_BANNERS_LANGUAGE_JOB_SLEEP_COEFFICIENT);
    }

    /**
     * Конструктор для тестов
     */
    FillBannersLanguageJob(int shard, BannerTypedRepository bannerTypedRepository,
                           BannerRelationsRepository bannerRelationsRepository,
                           BannerCommonRepository bannerCommonRepository,
                           BannerTextExtractor bannerTextExtractor, PpcPropertiesSupport ppcPropertiesSupport,
                           QueryrecService queryrecService, DslContextProvider dslContextProvider,
                           ClientRepository clientRepository) {
        super(shard, bannerTypedRepository,
                bannerRelationsRepository,
                bannerCommonRepository,
                bannerTextExtractor, queryrecService, dslContextProvider, clientRepository);

        this.ppcPropertiesSupport = ppcPropertiesSupport;

        jobEnabledProperty = ppcPropertiesSupport.get(FILL_BANNERS_LANGUAGE_JOB_ENABLED);
        sleepCoefficientProperty = ppcPropertiesSupport.get(FILL_BANNERS_LANGUAGE_JOB_SLEEP_COEFFICIENT);
    }

    @Override
    public void execute() {
        int shard = getShard();

        for (int iter = 0; iter < ITERATIONS_PER_EXECUTE; iter++) {
            boolean jobEnabled = jobEnabledProperty.getOrDefault(false);
            if (!jobEnabled) {
                return;
            }

            Double sleepCoefficient = sleepCoefficientProperty.get();
            if (sleepCoefficient == null) {
                logger.warn("Sleep coefficient is not set");
                return;
            }

            AtomicBoolean noBannersToProcess = new AtomicBoolean(false);

            new RelaxedWorker(sleepCoefficient).runAndRelax(() -> {
                Long minBidWithProcessedLanguage = ppcPropertiesSupport.get(minBidWithProcessedLanguage(getShard()))
                        .getOrDefault(Long.MAX_VALUE);

                List<Long> bidsToProcess =
                        bannerCommonRepository.getBannerIdsLessThan(shard, minBidWithProcessedLanguage,
                                BANNERS_TO_PROCESS_PER_ITERATION);

                if (bidsToProcess.isEmpty()) {
                    logger.warn("No banners to process");
                    noBannersToProcess.set(true);
                    return;
                }

                processBanners(shard, bidsToProcess);
                updateProperty(shard, bidsToProcess);
            });

            if (noBannersToProcess.get()) {
                return;
            }
        }
    }

    private void updateProperty(int shard, Collection<Long> bidsToProcess) {
        StreamEx.of(bidsToProcess)
                .min(Long::compare)
                .ifPresentOrElse(
                        minBid -> ppcPropertiesSupport.set(minBidWithProcessedLanguage(shard).getName(),
                                minBid.toString()),
                        () -> logger.error("Something's wrong, can't update property"));
    }
}
