package ru.yandex.direct.jobs.banner.language;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.repository.BannersToFillLanguageQueueRepository;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.scheduler.Hourglass;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоба для заполнения сброшенных языков баннеров.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(seconds = 60 * 2 * 10),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(seconds = 60 * 2 * 10),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 60 * 2, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class FillClearedBannersLanguageJob extends AbstractFillBannersLanguageJob {

    private static final int MAX_ITERATIONS_PER_EXECUTE = 25;

    private final BannersToFillLanguageQueueRepository bannersToFillLanguageQueueRepository;

    @Autowired
    FillClearedBannersLanguageJob(BannerTypedRepository bannerTypedRepository,
                                  BannerRelationsRepository bannerRelationsRepository,
                                  BannerCommonRepository bannerCommonRepository,
                                  BannerTextExtractor bannerTextExtractor,
                                  QueryrecService queryrecService,
                                  DslContextProvider dslContextProvider,
                                  BannersToFillLanguageQueueRepository bannersToFillLanguageQueueRepository,
                                  ClientRepository clientRepository) {
        super(bannerTypedRepository, bannerRelationsRepository, bannerCommonRepository, bannerTextExtractor,
                queryrecService, dslContextProvider, clientRepository);

        this.bannersToFillLanguageQueueRepository = bannersToFillLanguageQueueRepository;
    }

    /**
     * Конструктор для тестов
     */
    FillClearedBannersLanguageJob(int shard, BannerTypedRepository bannerTypedRepository,
                                  BannerRelationsRepository bannerRelationsRepository,
                                  BannerCommonRepository bannerCommonRepository,
                                  BannerTextExtractor bannerTextExtractor,
                                  QueryrecService queryrecService,
                                  DslContextProvider dslContextProvider,
                                  BannersToFillLanguageQueueRepository bannersToFillLanguageQueueRepository,
                                  ClientRepository clientRepository) {
        super(shard, bannerTypedRepository, bannerRelationsRepository, bannerCommonRepository, bannerTextExtractor,
                queryrecService, dslContextProvider, clientRepository);

        this.bannersToFillLanguageQueueRepository = bannersToFillLanguageQueueRepository;
    }

    @Override
    public void execute() {
        int shard = getShard();

        int iter = 0;
        boolean lastIteration = false;

        while (iter++ < MAX_ITERATIONS_PER_EXECUTE && !lastIteration) {
            List<Long> bidsToProcess =
                    bannersToFillLanguageQueueRepository.peekItems(shard, BANNERS_TO_PROCESS_PER_ITERATION);

            if (bidsToProcess.size() < BANNERS_TO_PROCESS_PER_ITERATION) {
                lastIteration = true;
            }

            if (bidsToProcess.isEmpty()) {
                return;
            }

            processBanners(shard, bidsToProcess);
            bannersToFillLanguageQueueRepository.removeItems(shard, bidsToProcess);
        }
    }
}
