package ru.yandex.direct.jobs.bannerstorage;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.STATUS_DRAFT;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.STATUS_MODERATING;
import static ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate.New;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба обрабатывает performance креативы из bannerstorage'a
 * <p>
 * 1. Креативы, не привязанные к баннерам, возвращаем в статус Черновик
 * Это нужно потому, что мы не можем корректно обработать такие креативы в новой Модерации
 * (для отправки и для приёма вердикта нужно, чтобы с креативом был связан баннер)
 * 2. Креативы, которые зависли на модерации более чем на сутки, отправляем заново и пишем в лог
 * Тут аналогичная история: может случиться так, что креатив привязан к нескольким баннерам и отправляется
 * в Модерацию с одним из них. А потом этот баннер удаляется, и осиротевший вердикт не может примениться,
 * в результате креатив остаётся в статусе Sent.
 * <p>
 * Такие ситуации единичные во всей базе, но они могут периодически появляться.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_ELWOOD,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 600, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ProcessStaleBannerstorageCreativesJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(ProcessStaleBannerstorageCreativesJob.class);

    private final CreativeRepository creativeRepository;
    private final BannerStorageClient bannerStorageClient;

    @Autowired
    public ProcessStaleBannerstorageCreativesJob(CreativeRepository creativeRepository,
                                                 BannerStorageClient bannerStorageClient) {
        this.creativeRepository = creativeRepository;
        this.bannerStorageClient = bannerStorageClient;
    }

    @Override
    public void execute() {
        // Креативы, не привязанные к баннерам, возвращаем в статус Черновик
        var withoutBanners = creativeRepository.getModeratingBannerstorageCreativesWithoutBanners(getShard(), 100);
        if (!withoutBanners.isEmpty()) {
            logger.info("Found {} creatives without banners: {}", withoutBanners.size(), withoutBanners);
            //
            for (var creativeId : withoutBanners) {
                List<Creative> creatives = bannerStorageClient.getCreatives(singletonList(creativeId.intValue()));
                if (creatives.isEmpty()) {
                    logger.error("Creative {} was not found in bannerstorage", creativeId);
                } else {
                    Creative creative = creatives.get(0);
                    if (creative.getStatus() != STATUS_DRAFT) {
                        logger.info("Call bannerstorage requestEdit for creative {}", creativeId);
                        bannerStorageClient.requestEdit(creativeId.intValue());
                    } else {
                        logger.info("Creative is in STATUS_DRAFT status already");
                    }
                    //
                    logger.info("Set statusModerate='New' for creative: {}", creativeId);
                    creativeRepository.setStatusModerate(getShard(), singletonList(creativeId), New);
                }
            }
        }

        // Креативы, которые зависли на модерации более чем на сутки, отправляем заново и пишем в лог
        var staleInModeration = creativeRepository.getBannerstorageCreativesStaleInModeration(getShard(), 24, 100);
        if (!staleInModeration.isEmpty()) {
            logger.info("Found {} stale creatives: {}", staleInModeration.size(), staleInModeration);
            //
            for (var creativeId : staleInModeration) {
                var creatives = bannerStorageClient.getCreatives(singletonList(creativeId.intValue()));
                if (creatives.isEmpty()) {
                    logger.error("Creative {} was not found in bannerstorage", creativeId);
                } else {
                    Creative creative = creatives.get(0);
                    if (creative.getStatus() != STATUS_MODERATING && creative.getStatus() != STATUS_DRAFT) {
                        logger.error("Creative {} has unexpected status {}, skip it", creativeId, creative.getStatus());
                    } else {
                        logger.info("Resend creative {} to moderation", creativeId);
                        // Меняем статус сначала на Error, потом на Ready, чтобы стриггерить ess-rule
                        creativeRepository.setStatusModerate(getShard(), singletonList(creativeId),
                                PerfCreativesStatusmoderate.Error);
                        creativeRepository.sendCreativesToModeration(getShard(), singletonList(creativeId));
                    }
                }
            }
        }
    }
}
