package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.bannersystem.BannerSystemClient;
import ru.yandex.direct.bannersystem.BsExportTableClient;
import ru.yandex.direct.bannersystem.container.exporttable.HolidayInfoRecord;
import ru.yandex.direct.core.entity.timetarget.model.HolidayItem;
import ru.yandex.direct.core.entity.timetarget.service.ProductionCalendarProviderService;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.regions.Region.BY_REGION_ID;
import static ru.yandex.direct.regions.Region.KAZAKHSTAN_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.TURKEY_REGION_ID;
import static ru.yandex.direct.regions.Region.UKRAINE_REGION_ID;
import static ru.yandex.direct.regions.Region.UZBEKISTAN_REGION_ID;

/**
 * Синхронизация списка государственных праздников и переносов выходных
 * {@link ru.yandex.direct.dbschema.ppcdict.tables.GreatHolidays}, забираемых по ручке из БК.
 * Список будет использоваться для показа в интерфейсе и модификации работы скриптов
 * <p>
 * <p>
 * Синхронизация выполняется только раз в день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 2),
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 14 2 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class HolidaySynchronizerJob extends DirectJob {

    /**
     * Разрешенные для использования в справочнике праздников номера геоинформационных регионов.
     * При добавлении новых значения, пожалуйста, дублируйте их в perl-версии
     * Holidays::LOCAL_HOLIDAYS_SUPPORTED_REGIONS
     *
     * @see <a href="https://svn.yandex-team.ru/websvn/wsvn/direct/trunk/protected/Holidays.pm">Holidays</a>
     */
    private static final Set<Long> LOCAL_HOLIDAYS_SUPPORTED_REGIONS =
            Collections.unmodifiableSet(
                    Stream.of(RUSSIA_REGION_ID, UKRAINE_REGION_ID, KAZAKHSTAN_REGION_ID, BY_REGION_ID, TURKEY_REGION_ID,
                            UZBEKISTAN_REGION_ID).collect(toSet()));

    private static final Logger logger = LoggerFactory.getLogger(HolidaySynchronizerJob.class);

    private final ProductionCalendarProviderService calendarProviderService;
    private final BsExportTableClient bsExportTableClient;

    @Autowired
    public HolidaySynchronizerJob(
            BannerSystemClient bannerSystemClient,
            ProductionCalendarProviderService calendarProviderService) {
        this.calendarProviderService = calendarProviderService;
        this.bsExportTableClient = new BsExportTableClient(bannerSystemClient);
    }

    @Override
    public void execute() {
        syncHolidays();
    }

    private void syncHolidays() {
        logger.info("Will fetch data now");
        final List<HolidayInfoRecord> bsHolidayInfoTableRows = bsExportTableClient.getHolidayInfoTableRowsList();
        logger.trace("BannerSystem rows: {}", bsHolidayInfoTableRows);

        final List<HolidayInfoRecord> allowedBsTableRowsList = bsHolidayInfoTableRows.stream()
                .filter(row -> null != row.getUpdateTime())
                .filter(row -> LOCAL_HOLIDAYS_SUPPORTED_REGIONS.contains(row.getRegionID()))
                .collect(Collectors.toList());

        if (allowedBsTableRowsList.isEmpty()) {
            logger.info("Got absent response from bs");
            return;
        }

        logger.trace("BannerSystem rows filtered by allowed region IDs: {}", allowedBsTableRowsList);
        logger.info("Successfully imported {} Holidays from BannerSystem", allowedBsTableRowsList.size());

        final Set<HolidayItem> bsHolidayItems = toHolidayItemSet(allowedBsTableRowsList);
        final List<HolidayItem> allDirectHolidays = calendarProviderService.getAllHolidays();
        logger.trace("All Direct holidays: {}", allDirectHolidays);
        final LocalDate today = LocalDate.now();
        final Set<HolidayItem> futureDirectHolidaysToDelete = allDirectHolidays.stream()
                .filter(h -> h.getDate() != null && h.getDate().isAfter(today))
                .filter(h -> !bsHolidayItems.contains(h))
                .collect(toSet());
        logger.trace("All Direct holidays after '{}' not contained in BannerSystem table (will be deleted): {}",
                today,
                futureDirectHolidaysToDelete);

        // leave only new rows to save
        bsHolidayItems.removeAll(allDirectHolidays);
        logger.trace("Final list of holidays to save: {}", bsHolidayItems);

        int updatedHolidays = calendarProviderService.updateHolidays(bsHolidayItems);
        logger.info("{} holidays were added/updated", updatedHolidays);

        int deletedHolidays = calendarProviderService.deleteHolidays(futureDirectHolidaysToDelete);
        logger.info("{} holidays were deleted", deletedHolidays);
    }

    private Set<HolidayItem> toHolidayItemSet(List<HolidayInfoRecord> tableRowsList) {
        return tableRowsList.stream()
                .map(this::toHolidayItem)
                .collect(toSet());
    }

    private HolidayItem toHolidayItem(HolidayInfoRecord row) {
        return new HolidayItem(
                row.getRegionID(),
                row.getUpdateTime().toInstant().atZone(ZoneId.systemDefault()).toLocalDate(),
                toHolidayItemType(row.getVirtualWeekday()));
    }

    HolidayItem.Type toHolidayItemType(HolidayInfoRecord.VirtualWeekdayType type) {
        return HolidayItem.Type.valueOf(type.name());
    }

}
