package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.core.entity.page.repository.PageRepository;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.core.entity.pages.model.PageOption;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Импорт из БК списка всех площадок.
 * Из YT таблицы PAGE_BS выгружает список всех площадок и обновляет их в таблице PAGES в PPCDICT.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        needCheck = TypicalEnvironment.class,
        //PRIORITY: спросить у хрустяшко про важность импорта пейджей
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ImportPagesJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ImportPagesJob.class);

    private static final String TABLE_PATH = "//home/yabs/dict/Page";
    private static final int COUNT_ITEMS_IN_STEP = 2_000;

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final PageRepository pageRepository;

    @Autowired
    public ImportPagesJob(YtProvider ytProvider, @Qualifier(DEFAULT_YT_CLUSTER) YtCluster ytCluster,
                          PageRepository pageRepository) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.pageRepository = pageRepository;
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        YtTable inputTable = new YtTable(TABLE_PATH);

        if (!ytOperator.exists(inputTable)) {
            logger.error("Table {} doesn't exists", inputTable);
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return;
        }

        final AtomicInteger changedCount = new AtomicInteger();
        ytOperator.readTableSnapshot(inputTable, new PageTableRow(), ImportPagesJob::convertRowToPage,
                r -> changedCount.addAndGet(saveChanges(r)), COUNT_ITEMS_IN_STEP);
        logger.info("count of changed pages: {}", changedCount.get());
    }

    int saveChanges(Collection<Page> pagesPart) {
        pagesPart = getOnlyChanged(pagesPart);
        if (pagesPart.isEmpty()) {
            return 0;
        }
        logger.debug("changed pages: {}", pagesPart);
        pageRepository.updatePages(pagesPart);
        return pagesPart.size();
    }

    /**
     * Возвращает список Page которых - нет в таблице PAGES в PpcDict, либо они отличаются
     */
    private Collection<Page> getOnlyChanged(Collection<Page> pagesFromYt) {
        if (pagesFromYt.isEmpty()) {
            return pagesFromYt;
        }
        List<Long> pagesId = pagesFromYt.stream()
                .map(Page::getId)
                .collect(Collectors.toList());
        Set<Page> pagesPpcDict = pageRepository.getPagesByIds(pagesId);
        return pagesFromYt.stream()
                .filter(Predicate.not(pagesPpcDict::contains))
                .collect(Collectors.toList());
    }

    private static Page convertRowToPage(PageTableRow row) {
        return createPage(
                row.getPageId(),
                row.getOrigPageId(),
                row.getName(),
                row.getDescription(),
                row.getTargetType(),
                row.getOptYaGroup(),
                row.getOptYaPage(),
                row.getOptDistribAdv());
    }

    static Page createPage(long ytPageId, long ytOrigPageId, String ytName, @Nullable String ytDescr, long ytTargetType,
                           boolean ytYaGroup, boolean ytYaPage, boolean ytDistribAdv) {
        String domain = convertDomain(ytName);
        String name = ytYaGroup ? "Яндекс" : domain;
        String groupNick = (ytYaGroup ? "yandex" : domain) + "." + (ytTargetType == 3L ? 3 : 0);
        int sorting = ytYaGroup ? 0 : 9;
        /*
        origPageId нужен для поиска настроек площадки по другим таблицам, например в ppcdict.page_place
        Для большинства площадок приходит OrigPageID=PageID, для некоторых OrigPageID=0 или OrigPageID != PageID
        OrigPageID=0 означает, что надо искать по PageID настройки. У нас будем вместо 0 сохранять PageID, чтобы
        можно было искать настройки площадки по origPageId не проверяя на 0
         */
        long origPageId = ytOrigPageId == 0 ? ytPageId : ytOrigPageId;
        Set<PageOption> options = getOptions(ytYaPage, ytDistribAdv);

        return new Page()
                .withId(ytPageId)
                .withOrigPageId(origPageId)
                .withDomain(domain)
                .withDescription(ytDescr)
                .withTargetType((short) ytTargetType)
                .withName(name)
                .withGroupNick(groupNick)
                .withSorting((short) sorting)
                .withOptions(options);
    }

    private static Set<PageOption> getOptions(boolean ytYaPage, boolean ytDistribAdv) {
        boolean isInternalAdPage = ytYaPage || ytDistribAdv;
        return isInternalAdPage ? Set.of(PageOption.INTERNAL_AD) : Collections.emptySet();
    }

    /**
     * Возвращает Host из {@param name} текста (если Host в нем есть)
     * Пример: https://play.google.com/store -> play.google.com
     */
    static String convertDomain(final String name) {
        String str = name.trim().split(" ")[0].toLowerCase();
        str = str.startsWith("http://") || str.startsWith("https://") ? str : "http://" + str;
        try {
            URI uri = new URI(str);
            return uri.getHost() != null ? uri.getHost() : name.toLowerCase();
        } catch (URISyntaxException e) {
            return name.toLowerCase();
        }
    }
}
