package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.time.LocalDate;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.core.entity.sspplatform.container.SspInfo;
import ru.yandex.direct.core.entity.sspplatform.repository.SspInfoYtRepository;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Синхронизация списка SSP-площадок в таблице {@link ru.yandex.direct.dbschema.ppcdict.tables.SspPlatforms}.
 * Актуальный список получаем из БК. Сохраняем в базе только те площадки, у которых в Options написано 'production'.
 * В этой задаче список только пополняется, очистка базы от отсутствующих в БК данных не нужна.
 * <p>
 * Синхронизация выполняется только раз в день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3), tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS,
        JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "39 1 * * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class SspPlatformSynchronizerJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(SspPlatformSynchronizerJob.class);

    private static final boolean FORCE_RUN_DEFAULT = false;
    static final PpcPropertyName<LocalDate> LAST_RUN_DATE_PROPERTY =
            new PpcPropertyName<>("ssp_platforms_last_run_date", PpcPropertyType.LOCAL_DATE);

    private final SspInfoYtRepository ytRepository;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public SspPlatformSynchronizerJob(SspInfoYtRepository ytRepository,
                                      PpcPropertiesSupport ppcPropertiesSupport,
                                      SspPlatformsRepository sspPlatformsRepository) {
        this.ytRepository = ytRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.sspPlatformsRepository = sspPlatformsRepository;
    }

    @Override
    public void execute() {
        syncSspPlatformsIfNecessary(FORCE_RUN_DEFAULT);
    }

    /**
     * @param forceRun - выполнить синхронизацию, даже если она уже была сегодня выполнена успешно.
     */
    public void syncSspPlatformsIfNecessary(boolean forceRun) {
        if (!canRun(forceRun)) {
            logger.info("SSP-platfroms already synced today");
            return;
        }

        syncSspPlatforms();

        updateLastRunProp();
    }

    private void syncSspPlatforms() {
        logger.info("Will fetch data now");
        List<SspInfo> sspPlatformsListAll = ytRepository.getAll();
        logger.info("Successfully imported {} SSP-platforms from BannerSystem", sspPlatformsListAll.size());

        List<SspInfo> sspPlatformsList = sspPlatformsListAll.stream()
                .filter(SspInfo::isOptionsProduction)
                .collect(Collectors.toList());
        logger.info("Got {} production SSP-platforms from BannerSystem", sspPlatformsList.size());

        sspPlatformsRepository.mergeSspPlatforms(sspPlatformsList);
        logger.info("Merged SSP-platforms");
    }

    boolean canRun(boolean forceRun) {
        if (!forceRun) {
            LocalDate lastRunDate = ppcPropertiesSupport.get(LAST_RUN_DATE_PROPERTY).get();
            if (lastRunDate != null && lastRunDate.compareTo(LocalDate.now()) >= 0) {
                return false;
            }
        }
        return true;
    }

    void updateLastRunProp() {
        LocalDate propValue = LocalDate.now();
        logger.info("Setting \"{}\" property to \"{}\"", LAST_RUN_DATE_PROPERTY, propValue);
        ppcPropertiesSupport.get(LAST_RUN_DATE_PROPERTY).set(propValue);
    }
}
