package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.bannersystem.BannerSystemClient;
import ru.yandex.direct.bannersystem.BsExportTableClient;
import ru.yandex.direct.bannersystem.model.exporttable.UniwordRecord;
import ru.yandex.direct.core.entity.stopword.repository.StopWordRepository;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Обновление стоп-слов из БК.
 * <p>
 * Если список изменился, то он перезаписывается в базу
 */
@Component
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3), tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 3600)
@ParametersAreNonnullByDefault
public class StopWordsDownloader extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(StopWordsDownloader.class);

    private final StopWordRepository stopWordsRepository;
    private final BsExportTableClient bsExportTableClient;

    @Autowired
    public StopWordsDownloader(StopWordRepository stopWordsRepository, BannerSystemClient bsClient) {
        this.stopWordsRepository = stopWordsRepository;
        this.bsExportTableClient = new BsExportTableClient(bsClient);
    }

    @Override
    public void execute() {
        List<UniwordRecord> uniwords = bsExportTableClient.getUniwordTableRows();

        Set<String> actualStopWords = StreamEx.of(uniwords)
                .filter(UniwordRecord::isStopWord)
                .map(UniwordRecord::getStopWord)
                .toSet();

        if (actualStopWords.isEmpty()) {
            logger.error("Got unexpected empty list of stop words");
            setJugglerStatus(JugglerStatus.CRIT, "Got unexpected empty list of stop words");
            return;
        }

        boolean isReplaced = stopWordsRepository.replaceStopWords(actualStopWords);
        if (isReplaced) {
            logger.info("Wrote {} new stop words {}", actualStopWords.size(), StringUtils.join(actualStopWords, ";"));
        } else {
            logger.info("The aren't new stop words");
        }

        String jugglerDescription = String.format("%d stop words fetched, list in db %s", actualStopWords.size(),
                isReplaced ? "updated" : "is actual");
        setJugglerStatus(JugglerStatus.OK, jugglerDescription);
    }
}
