package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.GenocideLogTransferParameter;
import ru.yandex.direct.jobs.configuration.GenocideLogsTransferParametersSource;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.ytwrapper.YtUtils.FASTBONE;

/**
 * Копирует таблицу с логом геноцида между кластерами с помощью Transfer Manager
 * Это нужно для быстрого доступа ко всем версиям таблиц с логами
 * Джоба, копирующая внутри кластера - {@link TransferGenocideLogInsideClusterJob}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2), needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.GROUP_INTERNAL_SYSTEMS})
@Hourglass(periodInSeconds = 900, needSchedule = ProductionOnly.class)
@ParameterizedBy(parametersSource = GenocideLogsTransferParametersSource.class)
@ParametersAreNonnullByDefault
public class TransferGenocideLogCrossClusterJob extends DirectParameterizedJob<GenocideLogTransferParameter> {

    private static final Logger logger = LoggerFactory.getLogger(TransferGenocideLogCrossClusterJob.class);

    private static final Duration COPYING_TIMEOUT = Duration.ofMinutes(90);

    private final GenocideLogsTransferParametersSource parametersSource;
    private final YtProvider ytProvider;
    private final TransferGenocideLogUtils utils;

    @Autowired
    public TransferGenocideLogCrossClusterJob(GenocideLogsTransferParametersSource parametersSource,
                                              YtProvider ytProvider,
                                              TransferGenocideLogUtils utils) {
        this.parametersSource = parametersSource;
        this.ytProvider = ytProvider;
        this.utils = utils;
    }

    @Override
    public void execute() {
        GenocideLogTransferParameter parameter = parametersSource.convertStringToParam(getParam());
        var sourceTable = utils.getSourceTable(parameter.getYabsCsKey());
        if (ytProvider.getOperator(parameter.getSourceCluster()).exists(sourceTable)) {
            utils.process(parameter, this::copyFunction);
        } else {
            utils.processDividedTables(parameter, this::concatenateAndCopyFunction);
        }
    }

    private void copyFunction(GenocideLogTransferParameter transferParameter) {

        YtCluster sourceCluster = transferParameter.getSourceCluster();
        YtCluster destinationCluster = transferParameter.getDestinationCluster();
        YPath sourcePath = utils.getSourceTable(transferParameter.getYabsCsKey()).ypath();
        YPath destinationPath = utils.getDestinationTable(transferParameter).ypath();

        Operation operation = ytProvider.get(destinationCluster).operations()
                .remoteCopyAndGetOp(sourcePath, destinationPath, sourceCluster.getName(), FASTBONE, true);
        logger.info("start copying, operation id = {}", operation.getId());
        operation.awaitAndThrowIfNotSuccess(COPYING_TIMEOUT);
    }

    private void concatenateAndCopyFunction(GenocideLogTransferParameter transferParameter) {
        var sourceCluster = transferParameter.getSourceCluster();
        var destinationCluster = transferParameter.getDestinationCluster();
        var sourceTables = utils.getDividedSourceTables(transferParameter.getYabsCsKey());
        var tempLocalDestination = YPath.simple(YtPathUtil.generateTemporaryPath());

        ytProvider.get(sourceCluster).cypress()
                .concatenate(Cf.toList(StreamEx.of(sourceTables).map(YtTable::ypath).toList()), tempLocalDestination);

        var tempRemoteDestination = YPath.simple(YtPathUtil.generateTemporaryPath());
        var operation = ytProvider.get(destinationCluster).operations()
                .remoteCopyAndGetOp(tempLocalDestination, tempRemoteDestination,
                        sourceCluster.getName(), FASTBONE, true);
        logger.info("start copying, operation id = {}", operation.getId());
        operation.awaitAndThrowIfNotSuccess(COPYING_TIMEOUT);

        var destinationPath = utils.getDestinationTable(transferParameter).ypath();
        ytProvider.get(destinationCluster).cypress()
                .move(tempRemoteDestination, destinationPath, true, true, false);
    }
}
