package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.GenocideLogTransferParameter;
import ru.yandex.direct.jobs.configuration.GenocideLogsTransferParametersSource;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Поддерживает самые свежие версии таблиц с логом геноцида внутри кластеров
 * Это нужно для быстрого доступа ко всем версиям таблиц с логами
 * Джоба, копирующая между кластерами - {@link TransferGenocideLogCrossClusterJob}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2), needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.GROUP_INTERNAL_SYSTEMS})
@Hourglass(periodInSeconds = 900, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class TransferGenocideLogInsideClusterJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(TransferGenocideLogInsideClusterJob.class);

    private final YtProvider ytProvider;
    private final TransferGenocideLogUtils utils;

    @Autowired
    public TransferGenocideLogInsideClusterJob(YtProvider ytProvider, TransferGenocideLogUtils utils) {
        this.ytProvider = ytProvider;
        this.utils = utils;
    }

    @Override
    public void execute() {
        List<GenocideLogTransferParameter> parameters = GenocideLogsTransferParametersSource.getInsideClusterParams();
        List<GenocideLogTransferParameter> failedParameters = new ArrayList<>();

        for (GenocideLogTransferParameter parameter : parameters) {
            try {
                var sourceTable = utils.getSourceTable(parameter.getYabsCsKey());
                if (ytProvider.getOperator(parameter.getSourceCluster()).exists(sourceTable)) {
                    utils.process(parameter, this::copyFunction);
                } else {
                    utils.processDividedTables(parameter, this::concatenateAndMoveFunction);
                }
            } catch (RuntimeException e) {
                logger.error(String.format("Got exception while processing parameter %s", parameter), e);
                failedParameters.add(parameter);
            }
        }

        if (!failedParameters.isEmpty()) {
            if (failedParameters.size() == parameters.size()) {
                throw new RuntimeException("All clusters processing failed");
            }
            setJugglerStatus(JugglerStatus.WARN, String.format("Failed to process parameter %s", failedParameters));
        }
    }

    private void copyFunction(GenocideLogTransferParameter transferParameter) {
        YtCluster sourceCluster = transferParameter.getSourceCluster();

        YtTable sourceTable = utils.getSourceTable(transferParameter.getYabsCsKey());
        YtTable destinationTable = utils.getDestinationTable(transferParameter);

        ytProvider.get(sourceCluster).cypress()
                .copy(sourceTable.ypath(), destinationTable.ypath(), true, true, false);
    }

    private void concatenateAndMoveFunction(GenocideLogTransferParameter transferParameter) {
        YtCluster sourceCluster = transferParameter.getSourceCluster();

        var sourceTables = utils.getDividedSourceTables(transferParameter.getYabsCsKey());
        var tempDestination = YPath.simple(YtPathUtil.generateTemporaryPath());
        var destinationTable = utils.getDestinationTable(transferParameter);
        var cypress = ytProvider.get(sourceCluster).cypress();
        cypress.concatenate(Cf.toList(StreamEx.of(sourceTables).map(YtTable::ypath).toList()), tempDestination);
        cypress.move(tempDestination, destinationTable.ypath(), true, true, false);
    }
}
