package ru.yandex.direct.jobs.bannersystem.dataimport;

import java.time.Instant;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jobs.configuration.GenocideLogTransferParameter;
import ru.yandex.direct.jobs.util.yt.YtEnvPath;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

@Component
public class TransferGenocideLogUtils {

    private static final Logger logger = LoggerFactory.getLogger(TransferGenocideLogUtils.class);

    private static final String OUTPUT_FOLDER = "import/genocide_results";
    private static final String GENOCIDE_RESULT_PATH = "//home/yabs-cs/key_%d/v2/export/genocide/genocide_results";
    private static final List<String> DIVIDED_GENOCIDE_RESULT_PATHS = List.of(
            "//home/yabs-cs/key_%d/v2/yabs_export/genocide/genocide_results",
            "//home/yabs-cs/key_%d/v2/bs_export/genocide/genocide_results");

    private static final String FRESHNESS_TIME_ATTRIBUTE = "freshness_time";

    private final YtProvider ytProvider;

    @Autowired
    public TransferGenocideLogUtils(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    YtTable getSourceTable(int yabsCsKey) {
        String inputPath = YtPathUtil.generatePath(String.format(GENOCIDE_RESULT_PATH, yabsCsKey));
        return new YtTable(inputPath);
    }

    YtTable getDestinationTable(GenocideLogTransferParameter parameter) {
        String outputPath = YtPathUtil.generatePath(
                ytProvider.getClusterConfig(parameter.getDestinationCluster()).getHome(),
                YtEnvPath.relativePart(),
                OUTPUT_FOLDER,
                parameter.getSourceCluster().name().toLowerCase() + '_' + parameter.getYabsCsKey());
        return new YtTable(outputPath);
    }

    /**
     * Сравниваем свежесть таблиц, применяем функцию копирования, ставим актуальный атрибут свежести
     *
     * @param transferParameter - параметры запуска
     * @param copyFunction      - функция для копирования таблиц
     */
    void process(GenocideLogTransferParameter transferParameter, Consumer<GenocideLogTransferParameter> copyFunction) {

        logger.info("start processing with parameter: {}", transferParameter);

        YtTable sourceTable = getSourceTable(transferParameter.getYabsCsKey());
        YtTable destinationTable = getDestinationTable(transferParameter);

        YtOperator destinationYtOperator = ytProvider.getOperator(transferParameter.getDestinationCluster());

        String modificationTime = ytProvider.getOperator(transferParameter.getSourceCluster())
                .readTableModificationTime(sourceTable);

        String ourFreshness = null;
        if (destinationYtOperator.exists(destinationTable)) {
            try {
                ourFreshness =
                        destinationYtOperator.readTableStringAttribute(destinationTable, FRESHNESS_TIME_ATTRIBUTE);
            } catch (Exception e) {
                logger.error("Error reading freshness_time attribute");
            }
        }

        logger.info("got source table modification_time = {}, destination table freshness_time = {}, " +
                        "source table: {}.`{}`, " +
                        "destination table: {}.`{}`",
                modificationTime, ourFreshness,
                transferParameter.getSourceCluster(), sourceTable.getPath(),
                transferParameter.getDestinationCluster(), destinationTable.getPath());

        if (ourFreshness != null && ourFreshness.compareTo(modificationTime) >= 0) {
            logger.info("don't need to copy");
            return;
        }

        copyFunction.accept(transferParameter);
        logger.info("copying done");

        destinationYtOperator.writeTableStringAttribute(destinationTable, FRESHNESS_TIME_ATTRIBUTE, modificationTime);
    }

    /**
     * Сравниваем свежесть таблиц, применяем функцию копирования, ставим актуальный атрибут свежести
     * В качестве нового атрибута свежести берём среднее арифметическое времени модификации исходных таблиц
     *
     * @param transferParameter - параметры запуска
     * @param copyFunction      - функция для копирования (слияния и перемещения) таблиц
     */
    void processDividedTables(GenocideLogTransferParameter transferParameter,
                             Consumer<GenocideLogTransferParameter> copyFunction) {
        logger.info("start processing with parameter: {}", transferParameter);

        var sourceTables = getDividedSourceTables(transferParameter.getYabsCsKey());
        var destinationTable = getDestinationTable(transferParameter);

        var destinationYtOperator = ytProvider.getOperator(transferParameter.getDestinationCluster());

        String modificationTime0 = ytProvider.getOperator(transferParameter.getSourceCluster())
                .readTableModificationTime(sourceTables.get(0));
        String modificationTime1 = ytProvider.getOperator(transferParameter.getSourceCluster())
                .readTableModificationTime(sourceTables.get(1));
        // среднее реагирует на изменение времени модификации каждой из таблиц. Не реагирует на микросекунды.
        long averageModificationTime = (Instant.parse(modificationTime0).toEpochMilli()
                + Instant.parse(modificationTime1).toEpochMilli()) / 2;
        var averageModificationTimeString = Instant.ofEpochMilli(averageModificationTime).toString();

        String ourFreshnessString = null;
        Long ourFreshness = null;
        if (destinationYtOperator.exists(destinationTable)) {
            try {
                ourFreshnessString = destinationYtOperator
                        .readTableStringAttribute(destinationTable, FRESHNESS_TIME_ATTRIBUTE);
                ourFreshness = Instant.parse(ourFreshnessString).toEpochMilli();
            } catch (Exception e) {
                logger.error("Error reading freshness_time attribute");
            }
        }

        logger.info("got source table modification_times = {} and {}, average = {}, " +
                        "destination table freshness_time = {}, " +
                        "source table: {}.`{}` and {}.`{}`, " +
                        "destination table: {}.`{}`",
                modificationTime0, modificationTime1, averageModificationTimeString, ourFreshnessString,
                transferParameter.getSourceCluster(), sourceTables.get(0).getPath(),
                transferParameter.getSourceCluster(), sourceTables.get(1).getPath(),
                transferParameter.getDestinationCluster(), destinationTable.getPath());

        // сравниваем с тем, что написали в прошлый раз
        if (ourFreshness != null && ourFreshness >= averageModificationTime) {
            logger.info("don't need to copy");
            return;
        }

        copyFunction.accept(transferParameter);
        logger.info("copying done");

        destinationYtOperator.writeTableStringAttribute(destinationTable, FRESHNESS_TIME_ATTRIBUTE,
                averageModificationTimeString);

    }

    /**
     * Получить пути до таблиц после разделения на сети/поиск
     *
     * @param yabsCsKey – контур
     * @return список таблиц
     */
    List<YtTable> getDividedSourceTables(int yabsCsKey) {
        return DIVIDED_GENOCIDE_RESULT_PATHS.stream()
                .map(path -> new YtTable(YtPathUtil.generatePath(String.format(path, yabsCsKey))))
                .collect(Collectors.toList());

    }
}
