package ru.yandex.direct.jobs.bannersystem.export.job;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.bannersystem.BannerSystemClient;
import ru.yandex.direct.bannersystem.BsImportAppStoreDataClient;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.bannersystem.export.container.MobileContentExportIndicators;
import ru.yandex.direct.jobs.bannersystem.export.service.BsExportMobileContentService;
import ru.yandex.direct.jobs.bannersystem.export.service.BsMobileContentExporter;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.libs.curator.lock.CuratorLock;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;

/**
 * Обновление в БК данных о мобильном контенте
 * <p>
 * Обновляет в БК данные о мобильном контенте (записи из таблицы mobile_content,
 * со statusBsSynced != "Yes", но не больше лимита, указанного в OBJECTS_PER_ITERATION).
 * <p>
 * Джоба работает до тех пор, пока не отправит все несинхронные данные из базы в БК, затем засыпает и запускается заново
 * в течение минуты
 * <p>
 * Данные об итерациях (отправлено/получено/ошибок/синхронизировано) отправляются в графит по следующему пути:
 * direct_one_min.db_configuration.production.flow.bsExportMobileContent.items_{sent,recieved,error,synced}.shard_$SHARD
 * <p>
 * Данные об итерациях отправляются:
 * - после первой отправки в БК;
 * - каждый FLUSH_INDICATORS_INTERVAL
 * - конце каждого цикла отправки, если они не были перед этим отправлены по одному из перечисленных выше условий
 */
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 15),
        needCheck = ProductionOnly.class,
        notifications = {
                @OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        //PRIORITY: отправляет информацию о мобильных приложения пользователя в БК, без этого не запустится РМП
        tags = {DIRECT_PRIORITY_0, GROUP_INTERNAL_SYSTEMS}
)
@ParametersAreNonnullByDefault
public class BsExportMobileContentJob extends DirectShardedJob {
    static final String SOLOMON_LABEL = "bsExportMobileContent";
    static final Duration LOCK_TIMEOUT = BsExportMobileContentService.DEFAULT_TIMEOUT.plusSeconds(120);
    static final String PROD_LOCK_NAME = "bsExportMobileContent";

    private static final int OBJECTS_PER_ITERATION = 5000;
    private static final Duration FLUSH_INDICATORS_INTERVAL = Duration.ofSeconds(60);
    private static final Logger logger = LoggerFactory.getLogger(BsExportMobileContentJob.class);
    private static final int MAX_ITERATIONS_NUM = 5;

    private final BsExportMobileContentService bsExportMobileContentService;
    private final SolomonPushClient solomonPushClient;
    private final BsImportAppStoreDataClient bsClient;
    private final MobileContentService mobileContentService;
    private final CuratorFrameworkProvider curatorFrameworkProvider;

    // Нам не очень интересно состояние лока до того как мы его реально получим, поэтому сразу ставим true
    private volatile boolean hasLock = true;

    @Autowired
    public BsExportMobileContentJob(
            BsExportMobileContentService bsExportMobileContentService,
            SolomonPushClient solomonPushClient,
            BannerSystemClient bsClient,
            MobileContentService mobileContentService,
            CuratorFrameworkProvider curatorFrameworkProvider) {
        this.bsExportMobileContentService = bsExportMobileContentService;
        this.solomonPushClient = solomonPushClient;
        this.bsClient = new BsImportAppStoreDataClient(bsClient);
        this.mobileContentService = mobileContentService;
        this.curatorFrameworkProvider = curatorFrameworkProvider;
    }

    BsExportMobileContentJob(int shard,
                             BsExportMobileContentService bsExportMobileContentService,
                             SolomonPushClient solomonPushClient,
                             BannerSystemClient bsClient,
                             MobileContentService mobileContentService,
                             CuratorFrameworkProvider curatorFrameworkProvider) {
        super(shard);
        this.bsExportMobileContentService = bsExportMobileContentService;
        this.solomonPushClient = solomonPushClient;
        this.bsClient = new BsImportAppStoreDataClient(bsClient);
        this.mobileContentService = mobileContentService;
        this.curatorFrameworkProvider = curatorFrameworkProvider;
    }

    @Override
    public void execute() {
        runIterationsInLock(getShard(), OBJECTS_PER_ITERATION, null, false);
    }

    /**
     * Запустить отправку данных в БК в ZooKeeper-локе, обеспечивающем уникальность экспорта.
     *
     * @param shard               шард
     * @param objectsPerIteration максимальное количество объектов в одном обращении в БК
     * @param mobileContentIds    список идентификаторов мобильного контента, который будем синхронизировать,
     *                            или null, если будем синхронизировать все несинхронные объекты
     * @param iterateOnce         провести не более одного обращения к БК, даже если данных больше,
     *                            чем влезает в один запрос к БК
     */
    void runIterationsInLock(int shard, int objectsPerIteration, @Nullable List<Long> mobileContentIds,
                             boolean iterateOnce) {
        // получаем ZooKeeper-блокировку для обеспечения уникальности экспорта мобильного контента
        // в перловой версии, если передавались конкретные mobileContentId, мы брали блокировки на каждый из них
        // но здесь мы этого делать не будем, так как боимся, что если идентификаторов будет слишком много,
        // ZooKeeper не выдержит
        try (CuratorLock ignore = curatorFrameworkProvider
                .getLock(String.format("%s_shard_%s", PROD_LOCK_NAME, shard), LOCK_TIMEOUT, this::lockIsLost)) {
            if (mobileContentIds == null) {
                mobileContentIds = bsExportMobileContentService.getMobileContentIdsForBsExport(shard);
            }
            if (hasLock) {
                runIterations(shard, objectsPerIteration, mobileContentIds, iterateOnce);
            }
        } catch (Exception e) {
            throw new RuntimeException("Iteration failed", e);
        }
    }

    void lockIsLost() {
        logger.error("We lost our lock, so job will continue to run until it reaches one of its breakpoint");
        hasLock = false;
    }

    /**
     * Запустить отправку данных в БК
     *
     * @param shard               шард
     * @param objectsPerIteration максимальное количество объектов в одном обращении в БК
     * @param mobileContentIds    список идентификаторов мобильного контента, который будем синхронизировать,
     *                            или null, если будем синхронизировать все несинхронные объекты
     * @param iterateOnce         провести не более одного обращения к БК, даже если данных больше,
     *                            чем влезает в один запрос к БК
     */
    void runIterations(int shard, int objectsPerIteration, List<Long> mobileContentIds, boolean iterateOnce) {
        LocalDateTime lastIndicatorsTransfer = LocalDateTime.MIN;
        List<MobileContentExportIndicators> collectedIndicators = new ArrayList<>();

        boolean continueIterating;
        int iterationNum = 1;
        do {
            continueIterating = false;
            logger.info("Start iteration #{}", iterationNum);

            BsMobileContentExporter exporter = getExporter(shard, objectsPerIteration, mobileContentIds);
            exporter.sendOneChunkOfMobileContent();
            logger.info("Iteration #{} end", iterationNum);

            collectedIndicators.add(exporter.getIndicators());
            if (lastIndicatorsTransfer.isBefore(LocalDateTime.now().minus(FLUSH_INDICATORS_INTERVAL))) {
                sendIndicators(shard, collectedIndicators);
                lastIndicatorsTransfer = LocalDateTime.now();
                collectedIndicators.clear();
            }

            iterationNum++;
            boolean needOneMoreIteration = exporter.isLimitExceeded();
            if (needOneMoreIteration && iterationNum <= MAX_ITERATIONS_NUM) {
                logger.info("Objects per iteration limit exceeded, will iterate once again");
                continueIterating = true;
            }
        } while (hasLock && !iterateOnce && continueIterating);

        if (!collectedIndicators.isEmpty()) {
            sendIndicators(shard, collectedIndicators);
        }
    }

    BsMobileContentExporter getExporter(int shard, int maxObjects, List<Long> mobileContentIds) {
        return new BsMobileContentExporter(shard, maxObjects, mobileContentIds, bsExportMobileContentService,
                mobileContentService, bsClient);
    }

    private void sendIndicators(int shard, List<MobileContentExportIndicators> collectedIndicators) {
        logger.info("Sending iteration indicators");
        MobileContentExportIndicators totalIndicators = new MobileContentExportIndicators();
        collectedIndicators.forEach(totalIndicators::addIndicators);

        var registry = SolomonUtils.newPushRegistry("flow", SOLOMON_LABEL, "shard", String.valueOf(shard));
        registry.gaugeInt64("items_sent_count").set(totalIndicators.getItemsSent());
        registry.gaugeInt64("items_received_count").set(totalIndicators.getItemsReceived());
        registry.gaugeInt64("items_error_count").set(totalIndicators.getItemsError());
        registry.gaugeInt64("items_synced_count").set(totalIndicators.getItemsSynced());
        try {
            solomonPushClient.sendMetrics(registry);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }
}
