package ru.yandex.direct.jobs.bannersystem.export.service;

import java.time.Duration;
import java.util.Arrays;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentForBsTransport;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;

import static ru.yandex.direct.multitype.entity.LimitOffset.limited;

/**
 * Объект, занимающийся экспортом в Баннерную Крутилку мобильного контента из базы Директа
 */
@ParametersAreNonnullByDefault
@Service
public class BsExportMobileContentService {
    public static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(120);
    private static final int CHANGE_STATUS_CHUNK_SIZE = 10000;

    private final MobileContentRepository mobileContentRepository;

    @Autowired
    public BsExportMobileContentService(MobileContentRepository mobileContentRepository) {
        this.mobileContentRepository = mobileContentRepository;
    }

    public List<Long> getMobileContentIdsForBsExport(int shard) {
        return mobileContentRepository
                .getMobileContentIdsWithStatusBsSynced(shard, Arrays.asList(StatusBsSynced.NO, StatusBsSynced.SENDING));
    }

    /**
     * Установить статус синхронизации с БК в Sending для объектов, у которых он No
     *
     * @param shard            шард
     * @param mobileContentIds список идентификаторов объектов, которым будем сбрасывать статус,
     *                         или null, если будем сбрасывать статус всем
     */
    void setStatusSending(int shard, List<Long> mobileContentIds) {
        setBsStatus(shard, StatusBsSynced.NO, StatusBsSynced.SENDING, mobileContentIds);
    }

    /**
     * Установить статус синхронизации с БК в Yes для объектов, у которых он Sending
     *
     * @param shard            шард
     * @param mobileContentIds список идентификаторов объектов, которым будем сбрасывать статус
     */
    int setStatusSynced(int shard, List<Long> mobileContentIds) {
        return setBsStatus(shard, StatusBsSynced.SENDING, StatusBsSynced.YES, mobileContentIds);
    }

    private int setBsStatus(int shard, StatusBsSynced from, StatusBsSynced to, List<Long> mobileContentIds) {
        int rowsChanged = 0;
        for (List<Long> idsChunk : Lists.partition(mobileContentIds, CHANGE_STATUS_CHUNK_SIZE)) {
            rowsChanged += mobileContentRepository.changeStatusBsSynced(shard, from, to, idsChunk);
        }

        return rowsChanged;
    }

    /**
     * Получить список объектов пригодных для экспорта мобильного контента в БК, но не более заданного лимита
     */
    List<MobileContentForBsTransport> getMobileContentForBsExport(int shard, int maxObjects,
                                                                  List<Long> mobileContentIds) {
        return mobileContentRepository.getMobileContentForBsExport(shard, mobileContentIds, limited(maxObjects));
    }
}
