package ru.yandex.direct.jobs.bannersystem.export.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bannersystem.BsImportAppStoreDataClient;
import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataIcon;
import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataItem;
import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataPrice;
import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataResponseItem;
import ru.yandex.direct.bannersystem.exception.BsImportAppStoreDataException;
import ru.yandex.direct.core.entity.mobilecontent.converter.MobileContentYtConverter;
import ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentAvatarSize;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentExternalWorldMoney;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentForBsTransport;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.mobilecontent.model.StoreActionForPrices;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.jobs.bannersystem.export.container.MobileContentExportIndicators;
import ru.yandex.direct.jobs.bannersystem.export.util.BsExportUtil;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.NumbersCheckUtils;
import ru.yandex.direct.utils.SystemUtils;

import static ru.yandex.direct.core.entity.mobilecontent.util.MobileContentUtil.getMoneyValue;
import static ru.yandex.direct.jobs.bannersystem.export.util.BsExportMobileContentUtil.getAgeLabelRepresentation;
import static ru.yandex.direct.jobs.bannersystem.export.util.BsExportMobileContentUtil.getOsTypeStringRepresentation;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, занимающийся экспортом в Баннерную Крутилку мобильного контента из базы Директа
 */
@ParametersAreNonnullByDefault
public class BsMobileContentExporter {
    private static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(120);

    private static final Logger logger = LoggerFactory.getLogger(BsMobileContentExporter.class);
    public static final String SCREENS_AVATAR_PREFIX = "//avatars.mds.yandex.net";
    public static final String SCREENS_PATH_FIELD = "path";

    private final BsExportMobileContentService service;
    private final MobileContentService mobileContentService;
    private final BsImportAppStoreDataClient bsClient;

    private final MobileContentExportIndicators indicators;
    private final UUID iterationUuid;

    private final List<Long> mobileContentIds;
    private final int maxObjects;
    private final int shard;

    private boolean limitExceeded;

    public BsMobileContentExporter(int shard, int maxObjects, List<Long> mobileContentIds,
                                   BsExportMobileContentService service, MobileContentService mobileContentService,
                                   BsImportAppStoreDataClient bsClient) {
        this.service = service;
        this.mobileContentService = mobileContentService;
        this.bsClient = bsClient;

        this.maxObjects = maxObjects;
        this.shard = shard;
        this.mobileContentIds = mobileContentIds;

        indicators = new MobileContentExportIndicators();
        iterationUuid = UUID.randomUUID();
        limitExceeded = false;
    }

    /**
     * Выполняет одну итерацию экспорта в БК данных о мобильном контенте. За одну итерацию производится не более одного
     * обращения к ручке импорта в БК
     */
    public void sendOneChunkOfMobileContent() {
        mobileContentIds.forEach(NumbersCheckUtils::checkIsValidId);

        try (TraceProfile ignore = Trace.current().profile("bs_export_mobile_content:one_bs_iteration")) {
            service.setStatusSending(shard, mobileContentIds);

            List<BsImportAppStoreDataItem> query = getExportQuery();

            if (!query.isEmpty()) {
                limitExceeded = query.size() >= maxObjects;
                indicators.addItemsSent(query.size());

                List<BsImportAppStoreDataResponseItem> response;
                try {
                    response = bsClient.sendAppStoreData(query, iterationUuid, DEFAULT_TIMEOUT);
                } catch (BsImportAppStoreDataException e) {
                    logError(e);
                    response = Collections.emptyList();
                }
                indicators.addItemsReceived(response.size());

                Set<Long> queryIds = listToSet(query, BsImportAppStoreDataItem::getMobileAppId);
                Set<Long> processedItemIds = parseBsResponse(queryIds, response);
                processBsResponse(processedItemIds);
            } else {
                logInfo("Nothing to send");
            }
        }
    }

    /**
     * Разобрать ответ БК, отделить успешно обработанные объекты от обработанных с ошибкой и залогировать все ошибки и
     * подозрительные данные
     *
     * @param requestIds набор идентификаторов объектов, которые были отправлены в запросе к БК
     * @param response   список ответов БК по каждому из отправленных объектов
     */
    Set<Long> parseBsResponse(Set<Long> requestIds, List<BsImportAppStoreDataResponseItem> response) {
        Set<Long> idsToSetStatus = new HashSet<>();

        try (TraceProfile ignore = Trace.current()
                .profile("bs_export_mobile_content:_parse_bs_response_import_application_store_data")) {
            Set<Long> processedItemIds = new HashSet<>();
            Set<Long> errorItemIds = new HashSet<>();
            for (BsImportAppStoreDataResponseItem item : response) {
                Long mobileContentId = item.getMobileAppId();

                if (item.getError() != null) {
                    // Получили ошибку в ответе
                    indicators.addItemsError(1);
                    if (mobileContentId != null) {
                        errorItemIds.add(mobileContentId);
                    }
                } else if (mobileContentId != null) {
                    // Считаем, что все хорошо и данные о контенте успешно синхронизированы
                    processedItemIds.add(mobileContentId);
                } else {
                    indicators.addItemsError(1);
                    logError(String.format("Strange entry in response: %s", JsonUtils.toJson(item)));
                }
            }

            // Смотрим, что совсем потеряли
            Set<Long> absentIds = new HashSet<>(requestIds);
            absentIds.removeAll(processedItemIds);
            absentIds.removeAll(errorItemIds);
            if (!absentIds.isEmpty()) {
                logError(String.format("absent mobile_content_ids in response: %s", absentIds));
            }

            if (!errorItemIds.isEmpty()) {
                logError(String.format("got error in response for mobile_content_ids: %s", errorItemIds));
            }

            // Вдруг нам туда лишнего накидали
            idsToSetStatus.addAll(processedItemIds);
            boolean hasExtraIds = idsToSetStatus.retainAll(requestIds);
            if (hasExtraIds) {
                processedItemIds.removeAll(idsToSetStatus);
                logError(String.format("got extra ids in response: %s", processedItemIds));
            }
        }

        return idsToSetStatus;
    }

    /**
     * Установить статус синхронизации в 'Yes' для обработанных объектов и залогировать все ошибки и подозрительные
     * данные
     *
     * @param processedItemIds набор идентификаторов объектов, которые были успешно обработаны в БК
     */
    void processBsResponse(Set<Long> processedItemIds) {
        try (TraceProfile ignore = Trace.current()
                .profile("bs_export_mobile_content:_process_bs_response_import_application_store_data")) {

            if (!processedItemIds.isEmpty()) {
                indicators.addItemsSynced(processedItemIds.size());
                int num = service.setStatusSynced(shard, new ArrayList<>(processedItemIds));

                logInfo(String.format("Successfully updated %s mobile content entries, set synced: %s",
                        processedItemIds.size(), num));
            }
        }
    }

    /**
     * Получить из базы данных список объектов мобильного контента в формате, принимаемом клиентом БК
     */
    List<BsImportAppStoreDataItem> getExportQuery() {
        List<MobileContentForBsTransport> snapshot;
        try (TraceProfile ignore = Trace.current().profile("bs_export_mobile_content:_get_snapshot")) {
            snapshot = service.getMobileContentForBsExport(shard, maxObjects, mobileContentIds);
        }
        return mapList(snapshot, this::convertDbToBsRepresentation);
    }

    /**
     * Сконвертировать описание объекта мобильного контента из внутреннего представления в представление БК
     */
    private BsImportAppStoreDataItem convertDbToBsRepresentation(MobileContentForBsTransport mobileContent) {
        BsImportAppStoreDataItem item = new BsImportAppStoreDataItem();
        item.setMobileAppId(mobileContent.getId());
        item.setIsAccessible(mobileContent.getIsAvailable());
        if (mobileContent.getAppSize() != null) {
            item.setAppSizeBytes(mobileContent.getAppSize().toBytes());
        }
        item.setOsType(getOsTypeStringRepresentation(mobileContent.getOsType()));
        item.setStoreName(mobileContentService.getStoreName(mobileContent.getOsType()));
        item.setStoreAppId(mobileContentService.getStoreAppId(mobileContent));
        item.setName(toExportName(mobileContent.getName()));
        item.setStoreContentId(mobileContent.getStoreContentId());

        item.setReviewCount(mobileContent.getRatingVotes());
        item.setReviewRating(mobileContent.getRating());

        item.setIcons(getIconsData(mobileContent));
        item.setPrices(getPricesData(mobileContent.getPrices()));
        item.setDownloadCount(ifNotNull(mobileContent.getDownloads(), downloads -> downloads.toString() + "+"));
        item.setScreens(ifNotNull(mobileContent.getScreens(), this::getScreensData));

        AgeLabel ageLabel = mobileContent.getAgeLabel();
        if (ageLabel != null) {
            item.setAgeLabel(getAgeLabelRepresentation(mobileContent.getAgeLabel()));
        }
        item.setStoreCountry(mobileContent.getStoreCountry());

        return item;
    }

    private List<Map<String, String>> getScreensData(List<Map<String, String>> screens) {
        return screens.stream().map(this::convertScreenItem).collect(Collectors.toList());
    }

    private Map<String, String> convertScreenItem(Map<String, String> stringStringMap) {
        return EntryStream.of(stringStringMap)
                .mapToValue((name, value) -> name.equals(SCREENS_PATH_FIELD) ? addAvatarHost(value) : value)
                .toMap();
    }

    private String addAvatarHost(String path) {
        Preconditions.checkArgument(path.startsWith("/"));
        return SCREENS_AVATAR_PREFIX + path;
    }

    /**
     * Получить из внутреннего представления мобильного контента представление цен действий в формате БК
     */
    Map<String, Map<String, BsImportAppStoreDataPrice>> getPricesData(
            Map<String, Map<StoreActionForPrices, MobileContentExternalWorldMoney>> prices) {
        Map<String, Map<String, BsImportAppStoreDataPrice>> result = new HashMap<>();
        for (Map.Entry<String, Map<StoreActionForPrices, MobileContentExternalWorldMoney>> storeCountryMapEntry : prices
                .entrySet()) {
            Map<String, BsImportAppStoreDataPrice> actionToPrice = new HashMap<>();

            for (Map.Entry<StoreActionForPrices, MobileContentExternalWorldMoney> actionMoneyEntry :
                    storeCountryMapEntry.getValue().entrySet()) {
                MobileContentExternalWorldMoney ewm = actionMoneyEntry.getValue();
                // В базе в поле для суммы может храниться null. В рамках транспорта в БК считаем, что это 0
                if (ewm.getSum() == null) {
                    ewm.setSum(BigDecimal.ZERO);
                }
                Money money = getMoneyValue(ewm);
                if (money == null) {
                    // не отправлям в БК цены для валют, неизвестных директу.
                    continue;
                }

                BsImportAppStoreDataPrice price;
                if (money.lessThanOrEqualEpsilon()) {
                    // для бесплатных приложений - валюта undef
                    price = new BsImportAppStoreDataPrice(BigDecimal.ZERO, null);
                } else {
                    price = new BsImportAppStoreDataPrice(money.bigDecimalValue(),
                            BsExportUtil.getBsIsoCurrencyCode(money.getCurrencyCode()));
                }

                actionToPrice.put(actionMoneyEntry.getKey().toString(), price);
            }

            if (!actionToPrice.isEmpty()) {
                result.put(storeCountryMapEntry.getKey(), actionToPrice);
            }
        }

        return result;
    }

    /**
     * Получить из внутреннего представления мобильного контента представление описания иконок контента в формате БК.
     * <p>
     * Если иконки еще не промодерированы, вернуть null
     */
    @Nullable
    Map<String, BsImportAppStoreDataIcon> getIconsData(MobileContentForBsTransport mobileContent) {
        if (mobileContent.getIconHash() == null || mobileContent.getStatusIconModerate() != StatusIconModerate.YES) {
            return null;
        }

        Map<String, BsImportAppStoreDataIcon> icons = new HashMap<>();
        for (MobileContentAvatarSize size : mobileContentService.getAvatarsSizes()) {
            if (!size.hasDimensions()) {
                continue;
            }
            icons.put(size.getName(), new BsImportAppStoreDataIcon(
                    mobileContentService
                            .generateUrlString(mobileContent.getOsType(), mobileContent.getIconHash(), size),
                    size.getWidth(), size.getHeight()
            ));
        }
        return icons;
    }

    /**
     * Если в названии приложения нет разрешённых символов оно заменяется на MobileContentYtConverter.UNDEFINED_NAME
     * При экспорте не хотим передавать такие названия¬
     */
    @Nullable
    private String toExportName(@Nullable String name) {
        if (MobileContentYtConverter.UNDEFINED_NAME.equals(name)) {
            return null;
        } else {
            return name;
        }
    }

    private void logError(Throwable e) {
        if (logger.isErrorEnabled()) {
            logger.error(String.format("%s\t%s", getPrefix(), e.getMessage()), e);
        }
    }

    private void logError(String message) {
        if (logger.isErrorEnabled()) {
            logger.error(String.format("%s\t%s", getPrefix(), message));
        }
    }

    private void logInfo(String message) {
        if (logger.isInfoEnabled()) {
            logger.info("{}\t{}", getPrefix(), message);
        }
    }

    private String getPrefix() {
        return String.format("[pid=%s,reqid=%s,shard=%s,uuid=%s]",
                SystemUtils.getPid(),
                Trace.current().getSpanId(),
                shard,
                iterationUuid);
    }

    public Boolean isLimitExceeded() {
        return limitExceeded;
    }

    public MobileContentExportIndicators getIndicators() {
        return indicators;
    }
}
