package ru.yandex.direct.jobs.bannersystem.logs;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.jobs.bannersystem.logs.configuration.BsExportLogsLogbrokerConsumerProperties;
import ru.yandex.direct.jobs.bannersystem.logs.model.BsExportLogsEntry;
import ru.yandex.direct.jobs.bannersystem.logs.reader.BsExportLogsLogbrokerReader;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.inside.yt.kosher.tables.YTableEntryTypes.YSON;

/**
 * Джоба для заливки логов запросов в БК в дин таблицу
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5), needCheck = TestingOnly.class, tags = {DIRECT_PRIORITY_1, GROUP_INTERNAL_SYSTEMS})
@HourglassDaemon(runPeriod = 7)
@Hourglass(needSchedule = TestingOnly.class, periodInSeconds = 7)
@ParametersAreNonnullByDefault
public class BsExportLogsLbToYtJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(BsExportLogsLbToYtJob.class);


    private static final int CHUNK_SIZE = 1_000;

    private final YtProvider ytProvider;
    private final YPath tablePath;
    private final YtCluster logsYtCluster;
    private final LogbrokerClientFactoryFacade logbrokerClientFactory;
    private final LogbrokerConsumerProperties consumerProperties;

    private LogbrokerBatchReader<BsExportLogsEntry> logbrokerReader;

    @Autowired
    public BsExportLogsLbToYtJob(YtProvider ytProvider, DirectConfig directConfig, TvmIntegration tvmIntegration) {
        this.ytProvider = ytProvider;

        DirectConfig config = directConfig.getBranch("bsexport_logs");

        this.tablePath = new YtTable(config.getString("table_path")).ypath();
        this.logsYtCluster = YtCluster.valueOf(config.getString("cluster"));

        Supplier<Credentials> credentialsSupplier = () -> {
            String serviceTicket = tvmIntegration.getTicket(TvmService.LOGBROKER_PRESTABLE);
            return Credentials.tvm(serviceTicket);
        };

        this.logbrokerClientFactory = new LogbrokerClientFactoryFacade(credentialsSupplier);

        String consumer = config.getString("consumer");
        this.consumerProperties = BsExportLogsLogbrokerConsumerProperties.toDynTableFromTesting()
                .setConsumer(consumer).build();
    }

    @Override
    public void initialize(TaskParametersMap taskParametersMap) {
        super.initialize(taskParametersMap);
        Supplier<SyncConsumer> syncConsumerSupplier = logbrokerClientFactory.createConsumerSupplier(consumerProperties);
        this.logbrokerReader = new RetryingLogbrokerBatchReader<>(
                () -> new BsExportLogsLogbrokerReader(syncConsumerSupplier));
    }

    @Override
    public void execute() {
        try {
            logbrokerReader.fetchEvents(this::processEvents);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private void processEvents(List<BsExportLogsEntry> rows) {
        logger.debug("Got {} rows from logbroker", rows.size());

        List<YTreeMapNode> ytRecords = StreamEx.of(rows)
                .map(BsExportLogsEntry::buildYTreeMapNode)
                .toList();

        Iterables.partition(ytRecords, CHUNK_SIZE).forEach(this::saveDataChunk);
    }

    private void saveDataChunk(List<YTreeMapNode> rows) {
        logger.debug("Table {}: inserting {} rows", tablePath, rows.size());
        Yt yt = ytProvider.get(logsYtCluster);
        yt.tables().insertRows(tablePath, false, false, YSON, Cf.wrap(rows).iterator());
    }

    @Override
    public void finish() {
        if (logbrokerReader != null) {
            try {
                logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
    }
}
