package ru.yandex.direct.jobs.base.dbqueuemonitoring

import ru.yandex.direct.dbqueue.DbQueueJobType
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.scheduler.support.DirectShardedJob
import ru.yandex.direct.solomon.SolomonUtils
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.time.Duration
import java.time.LocalDateTime
import java.util.concurrent.atomic.AtomicLong

abstract class BaseDbQueueMonitoringJob(
    private val dbQueueRepository: DbQueueRepository,
) : DirectShardedJob() {
    private val ageMinutes: AtomicLong = AtomicLong()
    private val queueSize: AtomicLong = AtomicLong()
    private val clientsWaiting: AtomicLong = AtomicLong()

    private lateinit var metricRegistry: MetricRegistry

    override fun execute() {
        metricRegistry = createMetricRegistry()
        ageMinutes.set(getAgeMinutes())
        queueSize.set(getQueueSize())
        clientsWaiting.set(getClientsCount())
    }

    abstract fun dbQueueJobType(): DbQueueJobType<*, *>
    abstract fun typeLabel(): String

    private fun getAgeMinutes(): Long {
        val now = LocalDateTime.now()

        val minCreateTime: LocalDateTime = dbQueueRepository
            .getMinCreateTimeByJobType(shard, dbQueueJobType())
            ?: return 0

        return Duration.between(minCreateTime, now).toMinutes()
    }

    private fun getQueueSize() =
        dbQueueRepository.getQueueSizeByJobType(shard, dbQueueJobType())

    private fun getClientsCount() =
        dbQueueRepository.getClientsCountByJobType(shard, dbQueueJobType())

    private fun createMetricRegistry(): MetricRegistry {
        val labels = Labels.of("type", typeLabel(), "shard", shard.toString())
        val metricRegistry = SolomonUtils.SOLOMON_REGISTRY.subRegistry(labels)

        metricRegistry.lazyGaugeInt64("age_minutes", ageMinutes::get)
        metricRegistry.lazyGaugeInt64("queue_size", queueSize::get)
        metricRegistry.lazyGaugeInt64("clients_waiting", clientsWaiting::get)

        return metricRegistry
    }

    override fun finish() {
        SolomonUtils.SOLOMON_REGISTRY.removeSubRegistry(metricRegistry.commonLabels)
    }
}
