package ru.yandex.direct.jobs.base.logdatatransfer;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.jobs.base.logdatatransfer.impl.VoidSourceDataPreprocessor;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static com.google.common.base.Preconditions.checkState;

/**
 * Джоба для перемещения данных из логов куда угодно с некоторой промежуточной трансформацией.
 * Подразумевается, что есть три сущности: логи, мета-информация (описывает, какие логи читать,
 * с какого момента, куда перекладывать), и целевая система (куда будут переложены данные).
 * <p>
 * Например, есть подневные логи просмотров видео-рекламы. Есть табличка ppc.video_segment_goals,
 * которая описывает, включен ли сбор сегментов аудиторий в группе, за какую дату последний раз
 * был обновлен сегмент и т.п. - это мета-информация. И есть сегмент аудитории в сервисе Яндекс.Аудитории -
 * это целевая система, куда загружаются извлеченные из логов uid'ы пользователей, видевших рекламу.
 *
 * @param <M> тип объекта метаинформации
 * @param <S> тип исходных данных из логов, один объект на все объекты с метаинформацией
 * @param <R> тип результата, передаваемого из стратегии обновления данных в целевой системе
 *            в стратегию обновления мета-информации, а так же в классы, обеспечивающие вычисление
 *            джагглерного статуса джобы, предоставляющие метрики и т.п.
 */
public class ShardedLogDataTransferJob<M, S, R> extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(ShardedLogDataTransferJob.class);

    private final MetaFetchingStrategy<M> metaFetchingStrategy;
    private final LogFetchingStrategy<M, S> logFetchingStrategy;
    private final SourceDataPreprocessor<M, S> sourceDataPreprocessor;
    private final TargetUpdatingStrategy<M, S, R> targetUpdatingStrategy;
    private final MetaUpdatingStrategy<M, S, R> metaUpdatingStrategy;

    private final JugglerStatusCalculator<S, R> jugglerStatusCalculator;
    private final MetricsProvider<R> metricsProvider;

    private final PpcProperty<Boolean> enableProperty;

    private final Supplier<Integer> limitSupplier;

    public ShardedLogDataTransferJob(MetaFetchingStrategy<M> metaFetchingStrategy,
                                     LogFetchingStrategy<M, S> logFetchingStrategy,
                                     @Nullable SourceDataPreprocessor<M, S> sourceDataPreprocessor,
                                     TargetUpdatingStrategy<M, S, R> targetUpdatingStrategy,
                                     MetaUpdatingStrategy<M, S, R> metaUpdatingStrategy,
                                     JugglerStatusCalculator<S, R> jugglerStatusCalculator,
                                     MetricsProvider<R> metricsProvider,
                                     Supplier<Integer> limitSupplier,
                                     @Nullable PpcProperty<Boolean> enableProperty) {
        this.metaFetchingStrategy = metaFetchingStrategy;
        this.logFetchingStrategy = logFetchingStrategy;
        this.sourceDataPreprocessor = sourceDataPreprocessor != null ?
                sourceDataPreprocessor :
                new VoidSourceDataPreprocessor<>();
        this.targetUpdatingStrategy = targetUpdatingStrategy;
        this.metaUpdatingStrategy = metaUpdatingStrategy;
        this.jugglerStatusCalculator = jugglerStatusCalculator;
        this.metricsProvider = metricsProvider;
        this.limitSupplier = limitSupplier;
        this.enableProperty = enableProperty;
    }

    public ShardedLogDataTransferJob(MetaFetchingStrategy<M> metaFetchingStrategy,
                                     LogFetchingStrategy<M, S> logFetchingStrategy,
                                     @Nullable SourceDataPreprocessor<M, S> sourceDataPreprocessor,
                                     TargetUpdatingStrategy<M, S, R> targetUpdatingStrategy,
                                     MetaUpdatingStrategy<M, S, R> metaUpdatingStrategy,
                                     JugglerStatusCalculator<S, R> jugglerStatusCalculator,
                                     MetricsProvider<R> metricsProvider,
                                     PpcProperty<Integer> limitProperty,
                                     int defaultLimit,
                                     @Nullable PpcProperty<Boolean> enableProperty) {
        this(metaFetchingStrategy,
                logFetchingStrategy,
                sourceDataPreprocessor,
                targetUpdatingStrategy,
                metaUpdatingStrategy,
                jugglerStatusCalculator,
                metricsProvider,
                () -> limitProperty.getOrDefault(defaultLimit),
                enableProperty);
    }

    /**
     * Алгоритм работы устроен следующим образом.
     * 1. В начале итерации стратегия получения мета-информации ({@link MetaFetchingStrategy})
     * возвращает порцию объектов, которые описывают, какие именно данные должны быть обработаны
     * (извлечены из логов и переложены в целевую систему).
     * 2. Полученный список объектов передается в стратегию извлечения данных из логов
     * ({@link LogFetchingStrategy}), а та в свою очередь по списку объектов с мета-информацией
     * возвращает объект с исходными данными.
     * 3. Исходные данные передаются в предобработчик {@link SourceDataPreprocessor}.
     * 4. Далее и мета-информация и полученные исходные данные передаются в стратегию обновления
     * данных в целевой системе ({@link TargetUpdatingStrategy}). Стратегия обновляет данные
     * в целевой системе и возвращает результат, на основании которого вычислитель джагглерного статуса
     * ({@link JugglerStatusCalculator}) и провайдер метрик ({@link MetricsProvider}) выполняют свои функции.
     * 4. Результат (из шага 4) передается в вычислитель джагглерного статуса ({@link JugglerStatusCalculator}),
     * чтобы в конце работы он вернул статус и комментарий для джобы.
     * Интерфейс поддерживает использование в цикле с последовательным добавлением результата.
     * 5. Результат (из шага 4) передается в провайдер метрик ({@link MetricsProvider}),
     * чтобы в конце работы он отправил необходимые метрики.
     * Интерфейс поддерживает использование в цикле с последовательным добавлением результата.
     * 7. После выполнения полезной работы выставляется джагглерный статус
     * и отправляются данные для построения графика.
     */
    @Override
    public final void execute() {

        if (!isEnabled()) {
            return;
        }

        logger.info("start execution at shard {}", getShard());

        int limit = limitSupplier.get();
        checkState(limit > 0, "limit must be greater than 0, but " + limit);

        List<M> meta = metaFetchingStrategy.fetch(getShard(), limit);
        S sourceData = logFetchingStrategy.fetch(meta);
        sourceData = sourceDataPreprocessor.preprocess(meta, sourceData);
        R result = targetUpdatingStrategy.updateTarget(meta, sourceData);
        metaUpdatingStrategy.updateMeta(getShard(), meta, sourceData, result);

        jugglerStatusCalculator.addResult(sourceData, result);
        metricsProvider.addResult(result);

        logger.info("execution finished at shard {}", getShard());

        setJugglerStatus(jugglerStatusCalculator.getJugglerStatus(), jugglerStatusCalculator.getDescription());
        metricsProvider.sendMonitoringInfo(getShard());
    }

    private boolean isEnabled() {
        if (enableProperty == null) {
            return true;
        }
        Boolean enabled = enableProperty.get();
        if (enabled == null) {
            logger.info("enableProperty is defined (name = '{}') but value in database is absent, stop working",
                    enableProperty.getName());
            return false;
        }

        logger.info("enableProperty is defined (name = '{}'), value in database = {}",
                enableProperty.getName(), enabled);
        return enabled;
    }
}
