package ru.yandex.direct.jobs.blackbox;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.blackbox.check.AudienceRobotPasswordRemainsCheckBundle;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.JugglerNumericEventsClient;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.misc.ip.IpAddress;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Проверка времени жизни пароля робота,
 * который используется для похода в Яндекс.Аудитории
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 24 * 60 * 60, needSchedule = NonDevelopmentEnvironment.class)
public class AudienceRobotPasswordExpireTimeJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(AudienceRobotPasswordExpireTimeJob.class);
    private static final String TRACE_BLACKBOX_USERINFO = "blackbox:userinfo";
    /**
     * Строгая паспортная политика = 3 месяца
     */
    private static final Duration PASSWORD_PERIOD = Duration.ofDays(90);
    /**
     * Код аттрибута, в котором хранится последняя дата обновления пароля
     * https://doc.yandex-team.ru/Passport/AuthDevGuide/concepts/DB_About.html#DB_About__db-attributes
     */
    private static final int PASSWORD_UPDATED_DATETIME_ATTRIBUTE = 20;
    private static final Duration EVENT_SEND_TIMEOUT = Duration.ofMinutes(10);

    private final BlackboxClient blackboxClient;
    private final String login;
    private final TvmService tvmService;
    private final TvmIntegration tvmIntegration;
    private final JugglerNumericEventsClient jugglerClient;
    private final AudienceRobotPasswordRemainsCheckBundle checkBundle;

    public AudienceRobotPasswordExpireTimeJob(
            BlackboxClient blackboxClient,
            @Value("${audience_client.login}") String login,
            TvmIntegration tvmIntegration,
            EnvironmentType environmentType,
            JugglerNumericEventsClient jugglerClient,
            AudienceRobotPasswordRemainsCheckBundle checkBundle
    ) {
        this.blackboxClient = blackboxClient;
        this.login = login;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
        this.jugglerClient = jugglerClient;
        this.checkBundle = checkBundle;
    }

    @Override
    public void execute() {
        var ipAddress = HttpUtil.getRemoteAddressForBlackbox();
        var tvmTicket = tvmIntegration.getTicket(tvmService);
        try {
            var passwordStartTime = getPasswordUpdatedTime(ipAddress, login, tvmTicket);
            var passwordEndTime = passwordStartTime.plus(PASSWORD_PERIOD);
            var remains = Duration.between(LocalDateTime.now(), passwordEndTime);
            sendPasswordRemainsDays(remains.toDays());
        } catch (RuntimeException e) {
            throw new RuntimeException("Failed to get password.update_time for '" + login + "'", e);
        }
    }

    private void sendPasswordRemainsDays(long remainsDays) {
        JugglerEvent jugglerEvent = checkBundle.generateEvent(remainsDays);
        logger.info("Sending juggler event to juggler: {}", jugglerEvent);
        jugglerClient.sendEvent(jugglerEvent, EVENT_SEND_TIMEOUT);
    }

    private LocalDateTime getPasswordUpdatedTime(IpAddress ipAddress, String login, String tvmTicket) {
        try (var ignored = Trace.current().profile(TRACE_BLACKBOX_USERINFO)) {
            var blackboxResponse = blackboxClient.userInfo(
                    ipAddress,
                    List.of(PASSWORD_UPDATED_DATETIME_ATTRIBUTE),
                    login,
                    tvmTicket
            );
            var passwordUpdatedTime = blackboxResponse.getAttributes().getOrThrow(PASSWORD_UPDATED_DATETIME_ATTRIBUTE);
            return DateTimeUtils.fromEpochSeconds(Long.parseLong(passwordUpdatedTime));
        }
    }
}
