package ru.yandex.direct.jobs.brandliftconditions;

import java.time.Duration;
import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBudgetReachDailyRepository;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.time.temporal.ChronoUnit.DAYS;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.RECALC_BRAND_LIFT_CAMPAIGNS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба чистки старых данных в campaigns_budget_reach
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_IVATKOEGOR,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 30 5 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class BrandLiftConditionsCleanerJob extends DirectShardedJob {
    private final CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository;
    private final DbQueueRepository dbQueueRepository;
    private final DirectConfig brandLiftConfig;
    private final ShardHelper shardHelper;

    @Autowired
    public BrandLiftConditionsCleanerJob(
            CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository,
            DbQueueRepository dbQueueRepository,
            DirectConfig directConfig,
            ShardHelper shardHelper) {
        this.campaignBudgetReachDailyRepository = campaignBudgetReachDailyRepository;
        this.dbQueueRepository = dbQueueRepository;
        this.brandLiftConfig = directConfig.getBranch("brand_lift");
        this.shardHelper = shardHelper;
    }

    @Override
    public void execute() {
        var lifeTimeThreshold = brandLiftConfig.getLong("life_time_threshold");
        var thresholdDate = LocalDate.now().minus(lifeTimeThreshold, DAYS);
        campaignBudgetReachDailyRepository.cleanOldRecords(getShard(), thresholdDate);

        for (var shard : shardHelper.dbShards()) {
            // удаляем все джобы на пересчёт, потому что в это же время запускается регулярный пересчёт
            dbQueueRepository.cleanup(shard, RECALC_BRAND_LIFT_CAMPAIGNS, Duration.ZERO, LimitOffset.maxLimited());
        }
    }
}
