package ru.yandex.direct.jobs.brandliftconditions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.brandlift.service.targetestimation.TargetEstimationsService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimation;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimationsDailyConsumer;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimationsTableRow;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static java.util.function.Function.identity;
import static ru.yandex.direct.feature.FeatureName.LOW_TOTAL_BUDGET_BRANDLIFT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба расчёта охвата и бюджета для brand lift
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_ANDREYPAV,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 30 5 * * ?", needSchedule = TypicalEnvironment.class)//каждый день в 05:30
@ParametersAreNonnullByDefault
public class BrandLiftConditionsCollectorJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(BrandLiftConditionsCollectorJob.class);
    private static final String BUDGET_ESTIMATIONS_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("brandliftconditions/budgetEstimations.sql").readLines());

    private final YtProvider ytProvider;
    private final ShardHelper shardHelper;
    private final DirectConfig brandLiftConfig;
    private final BudgetEstimationsDailyConsumer budgetEstimationsDailyConsumer;
    private final CampaignRepository campaignRepository;
    private final YtHelper ytHelper;
    private final TargetEstimationsService targetEstimationsService;
    private final CampaignBudgetReachDailyService campaignBudgetReachDailyService;
    private final FeatureService featureService;

    @Autowired
    public BrandLiftConditionsCollectorJob(
            YtProvider ytProvider,
            ShardHelper shardHelper,
            DirectConfig directConfig,
            BudgetEstimationsDailyConsumer budgetEstimationsDailyConsumer,
            CampaignRepository campaignRepository,
            YtHelper ytHelper,
            TargetEstimationsService targetEstimationsService,
            CampaignBudgetReachDailyService campaignBudgetReachDailyService,
            FeatureService featureService) {
        this.ytProvider = ytProvider;
        this.shardHelper = shardHelper;
        this.budgetEstimationsDailyConsumer = budgetEstimationsDailyConsumer;
        this.campaignRepository = campaignRepository;
        this.brandLiftConfig = directConfig.getBranch("brand_lift");
        this.ytHelper = ytHelper;
        this.targetEstimationsService = targetEstimationsService;
        this.campaignBudgetReachDailyService = campaignBudgetReachDailyService;
        this.featureService = featureService;
    }

    @Override
    public void execute() {
        var ytCluster = ytHelper.chooseFreshestCluster();
        if (ytCluster == null) {
            logger.error("Couldn't find available YT cluster, job will be skipped");
            return;
        }

        var ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);
        var budgetEstimationsTable = ytHelper.getBudgetEstimationsTable(ytCluster);

        try (var ignore = Trace.current().profile("brand_lift:yql", "budget_estimations")) {
            ytOperator.yqlExecute(
                    new YqlQuery(
                            BUDGET_ESTIMATIONS_QUERY,
                            ytHelper.getPool(ytCluster),
                            budgetEstimationsTable.getPath(),
                            StringUtils.join(shardHelper.dbShards(), ','),
                            null).withTitle("BrandLiftConditionsCollectorJob:budget_estimations_daily"));
            logger.info("Temp data about brand lift campaigns saved in {}", budgetEstimationsTable.getPath());
        }

        var chunkSize = brandLiftConfig.getBranch("yt").getInt("chunk_size");
        var targetThreshold = brandLiftConfig.getBranch("thresholds").getLong("target");

        for (long chunkIndex = 0; ; chunkIndex++) {
            ytOperator.readTableByRowRange(
                    budgetEstimationsTable,
                    budgetEstimationsDailyConsumer,
                    new BudgetEstimationsTableRow(),
                    chunkSize * chunkIndex,
                    chunkSize * (chunkIndex + 1));

            var budgetEstimationsDaily = budgetEstimationsDailyConsumer.getData();
            logger.info("Got {} brand lift campaigns from YT", budgetEstimationsDaily.size());
            if (budgetEstimationsDaily.isEmpty()) {
                break;
            }

            var campaignIds = mapList(budgetEstimationsDaily, BudgetEstimation::getCampaignId);
            var campaignsForShards = getCampaignsForShards(campaignIds);
            var campaigns = flatMap(campaignsForShards.values(), identity());
            var clientIds = mapList(campaigns, Campaign::getClientId);
            Map<ClientId, Boolean> clientsTotalBudgetWarnEnabled = featureService.isEnabledForClientIdsOnlyFromDb(
                    listToSet(clientIds, ClientId::fromLong),
                    LOW_TOTAL_BUDGET_BRANDLIFT.getName());

            var targetEstimations = targetEstimationsService.getTargetEstimations(campaigns);
            campaignBudgetReachDailyService.addCampaignDailyBudgets(
                    campaignsForShards, budgetEstimationsDaily, targetEstimations, targetThreshold,
                    clientsTotalBudgetWarnEnabled);

            budgetEstimationsDailyConsumer.clear();
        }
    }

    private Map<Integer, List<Campaign>> getCampaignsForShards(List<Long> campaignIds) {
        var campaignsForShards = new HashMap<Integer, List<Campaign>>();

        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .forEach((shard, campaignIdsOnShard) -> {
                    var campaignsOnShard = campaignRepository.getCampaigns(shard, campaignIdsOnShard);
                    campaignsForShards.put(shard, campaignsOnShard);
                });

        return campaignsForShards;
    }
}
