package ru.yandex.direct.jobs.brandliftconditions;

import java.time.Duration;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.brandlift.model.BrandLiftRecalcJobResult;
import ru.yandex.direct.core.entity.brandlift.service.targetestimation.TargetEstimationsService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimation;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimationsDailyConsumer;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimationsTableRow;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.feature.FeatureName.LOW_TOTAL_BUDGET_BRANDLIFT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба расчёта охвата и бюджета для brand lift по событию из ess + dbQueue
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_ANDREYPAV,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 30 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class BrandLiftConditionsDbQueueJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(BrandLiftConditionsDbQueueJob.class);
    private static final String BUDGET_ESTIMATIONS_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("brandliftconditions/budgetEstimations.sql").readLines());

    private final DbQueueRepository dbQueueRepository;
    private final YtHelper ytHelper;
    private final YtProvider ytProvider;
    private final CampaignRepository campaignRepository;
    private final TargetEstimationsService targetEstimationsService;
    private final CampaignBudgetReachDailyService campaignBudgetReachDailyService;
    private final BudgetEstimationsDailyConsumer budgetEstimationsDailyConsumer;
    private final DirectConfig brandLiftConfig;
    private final FeatureService featureService;

    public BrandLiftConditionsDbQueueJob(
            DbQueueRepository dbQueueRepository,
            YtHelper ytHelper,
            YtProvider ytProvider,
            CampaignRepository campaignRepository,
            TargetEstimationsService targetEstimationsService,
            CampaignBudgetReachDailyService campaignBudgetReachDailyService,
            BudgetEstimationsDailyConsumer budgetEstimationsDailyConsumer, DirectConfig directConfig,
            FeatureService featureService) {
        this.dbQueueRepository = dbQueueRepository;
        this.ytHelper = ytHelper;
        this.ytProvider = ytProvider;
        this.campaignRepository = campaignRepository;
        this.targetEstimationsService = targetEstimationsService;
        this.campaignBudgetReachDailyService = campaignBudgetReachDailyService;
        this.budgetEstimationsDailyConsumer = budgetEstimationsDailyConsumer;
        this.brandLiftConfig = directConfig.getBranch("brand_lift");
        this.featureService = featureService;
    }

    @Override
    public void execute() {
        var dbQueueLimit = brandLiftConfig.getBranch("db_queue").getInt("limit");
        var duration = Duration.ofMinutes(brandLiftConfig.getBranch("db_queue").getInt("duration_in_minutes"));
        var jobs = dbQueueRepository.grabBunchOfJobs(
                getShard(), DbQueueJobTypes.RECALC_BRAND_LIFT_CAMPAIGNS, duration, dbQueueLimit);

        if (jobs.isEmpty()) {
            logger.info("no tasks found for BrandLiftConditionsDbQueueJob-" + getShard());
            return;
        }

        var cids = jobs.stream().map(job -> job.getArgs().getCampaignId()).distinct().collect(toList());
        var ytCluster = ytHelper.chooseFreshestCluster();
        if (ytCluster == null) {
            logger.error("Couldn't find available YT cluster, job will be skipped");
            return;
        }

        var ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);
        var budgetEstimationsShardedTable = ytHelper.getBudgetEstimationsShardedTable(ytCluster, getShard());

        try (var ignore = Trace.current().profile("brand_lift:yql", "budget_estimations_" + getShard())) {
            ytOperator.yqlExecute(
                    new YqlQuery(
                            BUDGET_ESTIMATIONS_QUERY,
                            ytHelper.getPool(ytCluster),
                            budgetEstimationsShardedTable.getPath(),
                            String.valueOf(getShard()),
                            StringUtils.join(cids, ','))
                            .withTitle("BrandLiftConditionsDbQueueJob:budget_estimations_" + getShard()));
            logger.info("Temp data about brand lift campaigns saved in {}", budgetEstimationsShardedTable.getPath());
        }

        ytOperator.readTableByRowRange(
                budgetEstimationsShardedTable,
                budgetEstimationsDailyConsumer,
                new BudgetEstimationsTableRow(),
                0,
                dbQueueLimit);

        var budgetEstimations = budgetEstimationsDailyConsumer.getData();
        logger.info("Got {} brand lift campaigns from YT", budgetEstimations.size());

        if (!budgetEstimations.isEmpty()) {
            cids = mapList(budgetEstimations, BudgetEstimation::getCampaignId);
            var campaigns = campaignRepository.getCampaigns(getShard(), cids);
            var targetThreshold = brandLiftConfig.getBranch("thresholds").getLong("target");
            var targetEstimations = targetEstimationsService.getTargetEstimations(campaigns);
            var clientIds = mapList(campaigns, Campaign::getClientId);
            Map<ClientId, Boolean> clientsTotalBudgetWarnEnabled = featureService.isEnabledForClientIdsOnlyFromDb(
                    listToSet(clientIds, ClientId::fromLong),
                    LOW_TOTAL_BUDGET_BRANDLIFT.getName());

            campaignBudgetReachDailyService.addCampaignDailyBudgets(
                    Map.of(getShard(), campaigns), budgetEstimations, targetEstimations, targetThreshold,
                    clientsTotalBudgetWarnEnabled);
        }

        budgetEstimationsDailyConsumer.clear();

        for (var job : jobs) {
            dbQueueRepository.markJobFinished(getShard(), job, new BrandLiftRecalcJobResult(true));
        }
    }
}
