package ru.yandex.direct.jobs.brandliftconditions;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.brandlift.service.targetestimation.TargetEstimation;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStopReason;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignBudgetReachDaily;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBudgetReachDailyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jobs.brandliftconditions.budgetestimation.BudgetEstimation;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignBudgetReachDailyService {
    private static final Logger logger = LoggerFactory.getLogger(CampaignBudgetReachDailyService.class);

    private static final Integer WORK_DURATION_WITHOUT_LOW_REACH = 5;

    private static final Map<Currency, BrandLiftThreshold> THRESHOLD_CONSTANTS =
            Map.of(
                    Currencies.getCurrency(CurrencyCode.RUB),
                    new BrandLiftThreshold(30_000d, 14 * 30_000d, 63_000d, 900_000d),
                    Currencies.getCurrency(CurrencyCode.KZT),
                    new BrandLiftThreshold(40_000d, 14 * 40_000d, 96_300d, 1_350_000d),
                    Currencies.getCurrency(CurrencyCode.BYN),
                    new BrandLiftThreshold(190d, 14 * 190d, 526d, 7_650d),
                    Currencies.getCurrency(CurrencyCode.USD),
                    new BrandLiftThreshold(300d, 14 * 300d, 441d, 6_300d),
                    Currencies.getCurrency(CurrencyCode.EUR),
                    new BrandLiftThreshold(300d, 14 * 300d, 441d, 6_300d),
                    Currencies.getCurrency(CurrencyCode.CHF),
                    new BrandLiftThreshold(783d, 10_800d, 783d, 10_800d),
                    Currencies.getCurrency(CurrencyCode.TRY),
                    new BrandLiftThreshold(11_552d, 162_000d, 11_552d, 162_000d)
            );

    private final CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository;
    private final CampaignRepository campaignRepository;
    private final PpcPropertiesSupport ppcPropertySupport;


    public CampaignBudgetReachDailyService(
            CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository,
            CampaignRepository campaignRepository,
            PpcPropertiesSupport ppcPropertySupport) {
        this.campaignBudgetReachDailyRepository = campaignBudgetReachDailyRepository;
        this.campaignRepository = campaignRepository;
        this.ppcPropertySupport = ppcPropertySupport;
    }

    public static Double dailyThreshold(Currency currency,
                                        LocalDate brandSurveyBudgetDatePropVal,
                                        LocalDateTime createTime) {
        BrandLiftThreshold threshold = THRESHOLD_CONSTANTS.getOrDefault(currency,
                THRESHOLD_CONSTANTS.get(CurrencyCode.RUB));
        return  (brandSurveyBudgetDatePropVal != null
                && brandSurveyBudgetDatePropVal.atStartOfDay().isBefore(createTime)) ?
            threshold.getDailyThresholdNewAge() : threshold.getDailyThreshold();
    }

    public static Double totalThreshold(Currency currency,
                                        LocalDate brandSurveyBudgetDatePropVal,
                                        LocalDateTime createTime) {
        BrandLiftThreshold threshold = THRESHOLD_CONSTANTS.getOrDefault(currency,
                THRESHOLD_CONSTANTS.get(CurrencyCode.RUB));
        return  (brandSurveyBudgetDatePropVal != null
                && brandSurveyBudgetDatePropVal.atStartOfDay().isBefore(createTime)) ?
            threshold.getTotalThresholdNewAge() : threshold.getTotalThreshold();
    }

    public static List<CampaignBudgetReachDaily> createCampaignBudgetReachDaily(
            long targetThreshold,
            Map<Long, BudgetEstimation> budgetEstimationsMap,
            Map<Long, TargetEstimation> targetEstimationsMap,
            Map<Long, Integer> workDurations,
            List<Campaign> campaigns,
            boolean totalBudgetWarnEnabled,
            LocalDate brandSurveyBudgetDatePropVal,
            String brandSurveyId
    ) {

        var today = LocalDate.now();
        long targetForecast = campaigns.stream().map(campaign -> {
            var targetEstimation = targetEstimationsMap.get(campaign.getId());
            return targetEstimation != null ? targetEstimation.getTargetForecast() : 0L;
        }).reduce(Long::sum).orElse(0L);

        var brandSurveyStopReasons = EnumSet.noneOf(BrandSurveyStopReason.class);
        // DIRECT-119916: Не останавливать BrandLift после 5 дней
        var maxWorkDuration = campaigns.stream()
                .map(campaign -> workDurations.getOrDefault(campaign.getId(), 0))
                .max(Integer::compareTo)
                .orElse(0);
        if (targetForecast < targetThreshold && maxWorkDuration < WORK_DURATION_WITHOUT_LOW_REACH) {
            brandSurveyStopReasons.add(BrandSurveyStopReason.LOW_REACH);
        }

        var campaignBudgetEstimations = campaigns.stream()
                .collect(toMap(campaign -> campaign, campaign -> budgetEstimationsMap.get(campaign.getId())));
        LocalDateTime createTime = campaigns.stream()
                .map(Campaign::getCreateTime)
                .filter(Objects::nonNull)
                .findAny().orElse(LocalDateTime.now());

        for (var budgetEstimation : campaignBudgetEstimations.values()) {
            applyCurrencyRate(budgetEstimation);
        }

        double dailyThreshold = dailyThreshold(Currencies.getCurrency(campaignBudgetEstimations.get(campaigns.get(0)).getCurrency()),
                brandSurveyBudgetDatePropVal, createTime);
        double totalThreshold = totalThreshold(Currencies.getCurrency(campaignBudgetEstimations.get(campaigns.get(0)).getCurrency()),
                brandSurveyBudgetDatePropVal, createTime);
        var dailyBudget = getDailyBudget(campaignBudgetEstimations, today);

        if (dailyBudget < dailyThreshold) {
            brandSurveyStopReasons.add(totalBudgetWarnEnabled ? BrandSurveyStopReason.LOW_DAILY_BUDGET
                    : BrandSurveyStopReason.LOW_BUDGET);
        }

        var budgetTotalSpentEstimation = getBudgetTotalSpentEstimation(campaignBudgetEstimations, today);
        //прогнозные открутки за все время меньше, чем пороговое значение (для рублей - 420к)
        if (budgetTotalSpentEstimation < totalThreshold) {
            brandSurveyStopReasons.add(totalBudgetWarnEnabled ? BrandSurveyStopReason.LOW_TOTAL_BUDGET
                    : BrandSurveyStopReason.LOW_BUDGET);
        }

        var totalBudgetSpent = campaignBudgetEstimations.values().stream()
                .map(BudgetEstimation::getBudgetSpent)
                .reduce(Double::sum)
                .orElse(0.0);
        var totalBudgetEstimation = campaignBudgetEstimations.values().stream()
                .map(BudgetEstimation::getBudgetEstimated)
                .reduce(Double::sum)
                .orElse(0.0);

        return campaignBudgetEstimations.keySet().stream()
                .map(campaign -> {
                    var campaignId = campaign.getId();
                    var targetEstimation = targetEstimationsMap.get(campaignId);
                    var trafficLightColour = targetEstimation != null ? targetEstimation.getTrafficLightColour() : 0L;

                    return new CampaignBudgetReachDaily()
                            .withCampaignId(campaignId)
                            .withDate(today)
                            .withBudgetSpent(BigDecimal.valueOf(totalBudgetSpent))
                            .withBudgetEstimated(BigDecimal.valueOf(totalBudgetEstimation))
                            .withBudgetThreshold(BigDecimal.valueOf(dailyThreshold))
                            .withTargetForecast(targetForecast)
                            .withTrafficLightColour(trafficLightColour)
                            .withTargetThreshold(targetThreshold)
                            .withBrandSurveyStopReasons(brandSurveyStopReasons)
                            .withBrandSurveyId(brandSurveyId);
                })
                .collect(toList());
    }

    public void addCampaignDailyBudgets(
            Map<Integer, List<Campaign>> campaignsForShards,
            List<BudgetEstimation> budgetEstimationsDaily,
            List<TargetEstimation> targetEstimations,
            long targetThreshold,
            Map<ClientId, Boolean> clientsTotalBudgetWarnEnabled
    ) {
        var targetEstimationsMap = listToMap(targetEstimations, TargetEstimation::getCampaignId);
        var budgetEstimationsMap = listToMap(budgetEstimationsDaily, BudgetEstimation::getCampaignId);
        LocalDate brandSurveyBudgetDatePropVal = ppcPropertySupport.get(PpcPropertyNames.BRAND_SURVEY_BUDGET_DATE).get();

        for (var campaignWithShard : campaignsForShards.entrySet()) {
            try {
                var campaignIds = mapList(campaignWithShard.getValue(), Campaign::getId);

                var workDurations =
                        campaignBudgetReachDailyRepository.getBrandLiftWorkDuration(
                                campaignWithShard.getKey(), campaignIds);

                var campaignIdToBrandSurveyId = campaignRepository.getBrandSurveyIdsForCampaigns(
                        campaignWithShard.getKey(),
                        campaignWithShard.getValue().stream()
                                .map(Campaign::getId)
                                .collect(toList())
                );

                var campaignsByBrandSurvey = campaignWithShard.getValue().stream()
                        .collect(groupingBy(campaign -> campaignIdToBrandSurveyId.get(campaign.getId())));
                var campaignBudgetReachesDaily = campaignsByBrandSurvey
                        .entrySet()
                        .stream()
                        .map(campaignsWithBrandSurvey -> createCampaignBudgetReachDaily(
                                targetThreshold,
                                budgetEstimationsMap,
                                targetEstimationsMap,
                                workDurations,
                                campaignsWithBrandSurvey.getValue(),
                                clientsTotalBudgetWarnEnabled
                                        .getOrDefault(
                                                ClientId.fromLong(campaignsWithBrandSurvey.getValue().get(0).getClientId()),
                                                false
                                        ),
                                brandSurveyBudgetDatePropVal,
                                campaignsWithBrandSurvey.getKey()
                        ))
                        .flatMap(Collection::stream)
                        .collect(toList());

                campaignBudgetReachDailyRepository.addCampaignBudgetReaches(
                        campaignWithShard.getKey(), campaignBudgetReachesDaily);
            } catch (Exception ex) {
                logger.error("Got exception while inserting campaign daily budget reach records", ex);
            }
        }
    }

    private static double getDailyBudget(Map<Campaign, BudgetEstimation> campaignBudgetEstimations, LocalDate today) {
        return campaignBudgetEstimations.entrySet().stream()
                .map(pair -> getDailyBudget(pair.getKey(), pair.getValue(), today))
                .reduce(Double::sum).orElse(0.0);
    }

    private static double getDailyBudget(Campaign campaign, BudgetEstimation budgetEstimation, LocalDate today) {
        Long daysFromStart = calcDaysPeriod(getCampaignDate(campaign, true), today);
        if (campaign.getStatusShow() && daysFromStart > 1) {
            // если кампания уже показывается и стартовала раньше, чем вчера, смотрим реальные открутки.
            return budgetEstimation.getBudgetSpent();
        } else {
            // если кампания еще в черновике или стартовала недавно, смотрим прогноз откруток.
            return budgetEstimation.getBudgetEstimated();
        }
    }

    private static double getBudgetTotalSpentEstimation(Map<Campaign, BudgetEstimation> campaignBudgetEstimations,
                                                        LocalDate today) {
        return campaignBudgetEstimations.entrySet().stream()
                .map(pair -> getBudgetTotalSpentEstimation(pair.getKey(), pair.getValue(), today))
                .reduce(Double::sum).orElse(0.0);
    }

    private static double getBudgetTotalSpentEstimation(Campaign campaign, BudgetEstimation budgetEstimation,
                                                        LocalDate today) {
        var startDate = getCampaignDate(campaign, true);
        var daysToFinish = calcDaysPeriod(
                startDate.isAfter(today) ? startDate : today, getCampaignDate(campaign, false)
        );
        if (daysToFinish == null) {
            return 0d;
        }
        return budgetEstimation.getBudgetTotalSpent() + daysToFinish * budgetEstimation.getBudgetEstimated();
    }

    private static void applyCurrencyRate(BudgetEstimation budgetEstimation) {
        // только для тех валют, у которых нет своих порогов
        if (!THRESHOLD_CONSTANTS.containsKey(Currencies.getCurrency(budgetEstimation.getCurrency()))) {
            budgetEstimation.setBudgetEstimated(budgetEstimation.getBudgetEstimated() * budgetEstimation.getCurrencyRate());
            budgetEstimation.setBudgetSpent(budgetEstimation.getBudgetSpent() * budgetEstimation.getCurrencyRate());
            budgetEstimation.setBudgetTotalSpent(budgetEstimation.getBudgetTotalSpent() * budgetEstimation.getCurrencyRate());
        }
    }

    private static Long calcDaysPeriod(LocalDate dateFrom, @Nullable LocalDate dateTo) {
        if (dateTo == null) {
            //в cpm_default не работает BL сейчас
            return null;
        }

        return dateFrom.until(dateTo, ChronoUnit.DAYS);
    }

    private static LocalDate getCampaignDate(Campaign campaign, boolean start) {
        LocalDate date = start ? campaign.getStartTime() : campaign.getFinishTime();
        DbStrategy strategy = campaign.getStrategy();
        if (strategy != null && strategy.getStrategyData() != null) {
            StrategyName strategyName = strategy.getStrategyName();
            StrategyData strategyData = strategy.getStrategyData();
            if ((strategyName == StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD ||
                    strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD ||
                    strategyName == StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD) &&
                    (start || date == null || nvl(strategyData.getAutoProlongation(), 0L) == 0L)) {
                date = start ? strategyData.getStart() : strategyData.getFinish();
            }
        }

        return date;
    }

}
