package ru.yandex.direct.jobs.brandsafety;

import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.brandsafety.translation.BrandSafetyTranslations;
import ru.yandex.direct.core.entity.contentsegment.translation.ContentCategoryTranslations;
import ru.yandex.direct.core.entity.contentsegment.translation.ContentGenreTranslations;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.util.ReflectionTranslator;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Выгружает в YT все Brand Safety категории и жанры и категории контента в формате ID | translate(name, EN)
 */
@JugglerCheck(
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        ttl = @JugglerCheck.Duration(days = 2, hours = 1)
)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@SuppressWarnings({"squid:S1075", "squid:S00112"})
public class UploadBrandSafetyGoalsJob extends DirectParameterizedJob<YtCluster> {

    private static final String YQL_TEMPLATE =
            "INSERT INTO `%s` WITH TRUNCATE (id, parent_id, name, crypta_goal_type, bb_keyword, bb_keyword_value) VALUES %s";
    private static final String BRAND_SAFETY_YT_PATH = "export/brandsafety_categories";
    private static final int NUMBER_OF_PARAMETERS = 6;

    private static final Map<GoalType, TranslationBundle> TRANSLATION_BUNDLE_BY_GOAL_TYPE = Map.of(
            GoalType.BRANDSAFETY, BrandSafetyTranslations.INSTANCE,
            GoalType.CONTENT_CATEGORY, ContentCategoryTranslations.INSTANCE,
            GoalType.CONTENT_GENRE, ContentGenreTranslations.INSTANCE);

    private final YtProvider ytProvider;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final ReflectionTranslator reflectionTranslator;
    private final DirectExportYtClustersParametersSource parametersSource;

    public UploadBrandSafetyGoalsJob(YtProvider ytProvider,
                                     CryptaSegmentRepository cryptaSegmentRepository,
                                     ReflectionTranslator reflectionTranslator,
                                     DirectExportYtClustersParametersSource parametersSource) {
        this.ytProvider = ytProvider;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.reflectionTranslator = reflectionTranslator;
        this.parametersSource = parametersSource;
    }

    @Override
    public void execute() {
        var cluster = parametersSource.convertStringToParam(getParam());

        var yqlDataSource = ytProvider.getYql(cluster, YtSQLSyntaxVersion.SQLv1);
        try (var connection = yqlDataSource.getConnection()) {
            // Так как executeBatch не поддерживается драйвером YT, приходится делать вставку с помощью сбора
            // строки, которая потом станет prepared statement
            var interests = new ArrayList<>(cryptaSegmentRepository.getBrandSafetyAndContentSegments().values());

            if (interests.isEmpty()) {
                setJugglerStatus(JugglerStatus.WARN, "Content segments - Nothing to export");
                return;
            }

            var repeat = "(?,?,?,?,?,?),".repeat(interests.size());
            var questionMarks = repeat.substring(0, repeat.length() - 1);

            var home = ytProvider.getClusterConfig(cluster).getHome();
            var ytPath = YtPathUtil.generatePath(home, relativePart(), BRAND_SAFETY_YT_PATH);
            var yql = String.format(YQL_TEMPLATE, ytPath, questionMarks);

            try (var statement = connection.prepareStatement(yql)) {
                for (var i = 0; i < interests.size(); i++) {
                    var id = interests.get(i).getId();
                    var parentId = nvl(interests.get(i).getParentId(), 0L);
                    var cryptaGoalType = interests.get(i).getType().name();
                    var keyword = interests.get(i).getKeyword();
                    var keywordValue = interests.get(i).getKeywordValue();
                    var name = reflectionTranslator.translate(
                            interests.get(i).getTankerNameKey(),
                            TRANSLATION_BUNDLE_BY_GOAL_TYPE.get(interests.get(i).getType()),
                            Locale.ENGLISH
                    );

                    statement.setLong(i * NUMBER_OF_PARAMETERS + 1, id);
                    statement.setLong(i * NUMBER_OF_PARAMETERS + 2, parentId);
                    statement.setString(i * NUMBER_OF_PARAMETERS + 3, name);
                    statement.setString(i * NUMBER_OF_PARAMETERS + 4, cryptaGoalType);
                    if (keyword == null) {
                        statement.setNull(i * NUMBER_OF_PARAMETERS + 5, Types.VARCHAR);
                    } else {
                        statement.setString(i * NUMBER_OF_PARAMETERS + 5, keyword);
                    }

                    if (keywordValue == null) {
                        statement.setNull(i * NUMBER_OF_PARAMETERS + 6, Types.VARCHAR);
                    } else {
                        statement.setString(i * NUMBER_OF_PARAMETERS + 6, keywordValue);
                    }
                }
                statement.execute();
            }

            setJugglerStatus(
                    JugglerStatus.OK,
                    String.format("%s content segments exported to %s", interests.size(), cluster)
            );
        } catch (SQLException e) {
            setJugglerStatus(
                    JugglerStatus.CRIT,
                    String.format("Failed to export content segments to %s", cluster)
            );
            throw new RuntimeException(e);
        }
    }
}
