package ru.yandex.direct.jobs.bsclearidhistory;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.keyword.container.CampaignIdAndKeywordIdPair;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.statistics.container.AdGroupIdAndPhraseIdPair;
import ru.yandex.direct.core.entity.statistics.repository.BsAuctionStatRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.TimeProvider;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Очищает таблицы {@link ru.yandex.direct.dbschema.ppc.tables.BidsPhraseidHistory} от записей,
 * которые либо относятся к удаленным фразам, либо были отправлены в БК и устарели,
 * и {@link ru.yandex.direct.dbschema.ppc.tables.BsAuctionStat} от записей, которые не использовались последний день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 3), tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS})
@Hourglass(cronExpression = "0 59 3 * * ?", needSchedule = TypicalEnvironment.class)
public class BsClearIdHistoryJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(BsClearIdHistoryJob.class);

    /**
     * время устаревания данных в bs_auction_stat
     */
    private static final Duration STAT_TIME_EXPIRE = Duration.ofDays(1);
    /**
     * время устаревания данных в bids_phraseId_history для не архивных кампаний
     */
    private static final Duration BS_HISTORY_EXPIRE = Duration.ofHours(6);
    /**
     * время устаревания данных в bids_phraseId_history для архивных кампаний
     */
    private static final Duration BS_HISTORY_ARC_EXPIRE = Duration.ofDays(90);
    private static final int DELETE_FROM_PRIORITIES_LIMIT = 500_000;
    private static final int DELETE_FROM_BIDS_PHRASEID_HISTORY_CHUNK_SIZE = 10_000;
    private static final int DELETE_FROM_BS_AUCTION_STAT_CHUNK_SIZE = 1000;
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(0.3);


    private final BsAuctionStatRepository bsAuctionStatRepository;
    private final KeywordRepository keywordRepository;
    private final TimeProvider timeProvider;


    @Autowired
    public BsClearIdHistoryJob(BsAuctionStatRepository bsAuctionStatRepository, KeywordRepository keywordRepository) {
        this.bsAuctionStatRepository = bsAuctionStatRepository;
        this.keywordRepository = keywordRepository;
        this.timeProvider = new TimeProvider();
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда и timeProvider.
     */
    BsClearIdHistoryJob(int shard, BsAuctionStatRepository bsAuctionStatRepository,
            KeywordRepository keywordRepository, TimeProvider timeProvider)
    {
        super(shard);
        this.bsAuctionStatRepository = bsAuctionStatRepository;
        this.keywordRepository = keywordRepository;
        this.timeProvider = timeProvider;
    }

    @Override
    public void execute() {
        clearBidsPhraseIdHistory(getShard());
        clearUnusedIdsFromAuctionStat(getShard());
    }

    private void clearBidsPhraseIdHistory(int shard) {
        LocalDateTime historyArcExpireDateTime = timeProvider.now().minus(BS_HISTORY_ARC_EXPIRE);
        LocalDateTime historyExpireDateTime = timeProvider.now().minus(BS_HISTORY_EXPIRE);
        logger.info("Cleaning bids_phraseid_history with {} expireDateTime for archived campaigns and with {}" +
                " for not archived campaigns", historyArcExpireDateTime, historyExpireDateTime);

        List<CampaignIdAndKeywordIdPair> keys = new ArrayList<>();
        keys.addAll(keywordRepository.getIdsOfBidsPhraseIdHistoryForDeletedCampaigns(shard));
        keys.addAll(keywordRepository.getIdsOfBidsPhraseIdHistoryForArchivedCampaigns(shard, historyArcExpireDateTime));
        keys.addAll(keywordRepository.getIdsOfBidsPhraseIdHistoryForNotArchivedCampaigns(shard, historyExpireDateTime));

        logger.info("Fetched {} rows from bids_phraseid_history to delete", keys.size());
        int totalDeleted = 0;
        for (List<CampaignIdAndKeywordIdPair> chunk : Iterables
                .partition(keys, DELETE_FROM_BIDS_PHRASEID_HISTORY_CHUNK_SIZE)) {
            totalDeleted += RELAXED_WORKER.callAndRelax(() ->
                    keywordRepository.deleteFromBidsPhraseIdHistoryTable(shard, chunk));
        }
        logger.info("Totally deleted {} rows from bids_phraseid_history", totalDeleted);
    }

    private void clearUnusedIdsFromAuctionStat(int shard) {
        LocalDateTime statTimeLimit = timeProvider.now().minus(STAT_TIME_EXPIRE);
        logger.info("Cleaning bs_auction_stat with {} statTimeLimit", statTimeLimit);
        List<AdGroupIdAndPhraseIdPair> unusedIds;
        do {
            unusedIds = bsAuctionStatRepository.getUnusedIds(shard, statTimeLimit, DELETE_FROM_PRIORITIES_LIMIT);
            for (List<AdGroupIdAndPhraseIdPair> chunk : Iterables.partition(unusedIds,
                    DELETE_FROM_BS_AUCTION_STAT_CHUNK_SIZE)) {
                RELAXED_WORKER.runAndRelax(() -> {
                    int deleted = bsAuctionStatRepository.deleteUnusedByIds(shard, chunk, statTimeLimit);
                    logger.info("Deleted {} rows from bs_auction_stat", deleted);
                });
            }
        } while (unusedIds.size() == DELETE_FROM_PRIORITIES_LIMIT);
    }
}
