package ru.yandex.direct.jobs.bsexport;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

import javax.xml.soap.MessageFactory;
import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import com.google.protobuf.Any;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bannersystem.BannerSystemClient;
import ru.yandex.direct.bannersystem.BsUriFactory;
import ru.yandex.direct.bsexport.BsResponseProcessor;
import ru.yandex.direct.bsexport.iteration.BsExportIterationContext;
import ru.yandex.direct.bsexport.iteration.BsExportIterationFactory;
import ru.yandex.direct.bsexport.iteration.container.ExportCandidatesSelectionCriteria;
import ru.yandex.direct.bsexport.logbroker.FeedMessagesLogbrokerWriterFactory;
import ru.yandex.direct.bsexport.logbroker.FeedMessegesLogbrokerWriter;
import ru.yandex.direct.bsexport.messaging.SoapResponseDeserializer;
import ru.yandex.direct.bsexport.messaging.SoapSerializer;
import ru.yandex.direct.bsexport.model.FeedRequestMessage;
import ru.yandex.direct.bsexport.model.LogbrokerDebugInfo;
import ru.yandex.direct.bsexport.model.MessageType;
import ru.yandex.direct.bsexport.model.Order;
import ru.yandex.direct.bsexport.model.UpdateData2Request;
import ru.yandex.direct.bsexport.model.UpdateData2RequestSoapMessage;
import ru.yandex.direct.bsexport.model.UpdateData2Response;
import ru.yandex.direct.bsexport.query.order.OrderDataFactory;
import ru.yandex.direct.bsexport.snapshot.BsExportSnapshot;
import ru.yandex.direct.bsexport.snapshot.SnapshotDataFactory;
import ru.yandex.direct.bsexport.util.QueryComposer;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.SystemUtils;

// перенос на java заморожен/отменен, но код еще может пригодиться
public class BsExportWorker extends DirectParameterizedJob<BsExportWorkerParam> {
    private static final Logger logger = LoggerFactory.getLogger(BsExportWorker.class);

    private final BsExportIterationFactory iterationFactory;

    private final SnapshotDataFactory snapshotDataFetcher;
    private final BsExportWorkerParametersSource bsExportWorkerParametersSource;
    private final BannerSystemClient bannerSystemClient;
    private final FeedMessagesLogbrokerWriterFactory logbrokerWriterFactory;
    private FeedMessegesLogbrokerWriter logbrokerWriter;
    private final ShardHelper shardHelper;
    private final BsResponseProcessor bsResponseProcessor;
    private final Duration bsSoapTimeout;

    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");


    public BsExportWorker(BsExportIterationFactory iterationFactory,
                          SnapshotDataFactory snapshotDataFetcher,
                          BsExportWorkerParametersSource bsExportWorkerParametersSource,
                          FeedMessagesLogbrokerWriterFactory logbrokerWriterFactory,
                          ShardHelper shardHelper,
                          BannerSystemClient bannerSystemClient,
                          BsResponseProcessor bsResponseProcessor,
                          DirectConfig directConfig) {
        this.iterationFactory = iterationFactory;
        this.snapshotDataFetcher = snapshotDataFetcher;
        this.bsExportWorkerParametersSource = bsExportWorkerParametersSource;
        this.bannerSystemClient = bannerSystemClient;
        this.logbrokerWriterFactory = logbrokerWriterFactory;
        this.shardHelper = shardHelper;
        this.bsResponseProcessor = bsResponseProcessor;
        this.bsSoapTimeout = directConfig.getDuration("bs_export_worker.soap.timeout");
    }


    @Override
    public void execute() {

        var param = bsExportWorkerParametersSource.convertStringToParam(getParam());
        ExportCandidatesSelectionCriteria selectionCriteria = ExportCandidatesSelectionCriteria.builder()
                .setWorkerSpec(param.getWorkerSpec())
                .setLimit(30)
                .setLockNewCampaigns(true)
                .setShard(param.getShard())
                .build();

        try (BsExportIterationContext context = iterationFactory.lockCampaignsInQueue(selectionCriteria)) {
            iteration(context);
        }
    }


    private void iteration(BsExportIterationContext iterationContext) {
        BsExportSnapshot snapshot = snapshotDataFetcher.getSnapshot(iterationContext);
        logger.debug("snapshot fetched");

        OrderDataFactory orderDataFactory = new OrderDataFactory(snapshot);
        List<Order> orders = new ArrayList<>();
        for (Long campaignId : iterationContext.getDataCampaignIds()) {
            Order.Builder order = orderDataFactory.getOrder(campaignId);
            orders.add(order.build());
        }

        if (orders.isEmpty()) {
            return;
        }
        var requestUuid = UUID.randomUUID();
        sendLogbrokerQuery(requestUuid, iterationContext, orders);
        var response = sendSoapMessage(requestUuid, iterationContext, orders);
        var processedResponse = bsResponseProcessor.process(orders, response);
        iterationContext.delayCampaigns(processedResponse.getUndoneCampaigns());
        iterationContext.moveToBuggy(processedResponse.getErrorCampaigns());
        iterationContext.markCampaignsAsSynced(processedResponse.getProcessedCampaigns());
    }

    private void sendLogbrokerQuery(UUID requestUuid, BsExportIterationContext context, List<Order> orders) {
        var iterationId = shardHelper.generateBsExportIterId(1).get(0);
        LogbrokerDebugInfo.Builder logbrokerDebugInfoBuilder = LogbrokerDebugInfo.newBuilder();
        logbrokerDebugInfoBuilder.setHost(SystemUtils.hostname())
                .setReqid(Trace.current().getTraceId())
                .setShard(context.getShard())
                .setSendTime(DATE_TIME_FORMATTER.format(LocalDateTime.now()))
                .setParNormNick(getParNickName(context.getWorkerSpec()));
        List<FeedRequestMessage> feedRequestMessages = new ArrayList<>();
        for (var order : orders) {
            FeedRequestMessage.Builder messageBuilder = FeedRequestMessage.newBuilder();
            messageBuilder.setData(Any.pack(order))
                    .setCid(order.getEID())
                    .setFullExportFlag(context.getWorkerType() == WorkerType.FULL_LB_EXPORT ? 1 : 0)
                    .setLevel(MessageType.ORDER)
                    .setUuid(requestUuid.toString())
                    .setIterId(iterationId)
                    .setDebugInfo(logbrokerDebugInfoBuilder);
            feedRequestMessages.add(messageBuilder.build());
        }

        var logbrokerWriterLocal = getLogbrokerWriter(context.getShard(), context.getWorkerType(),
                context.getWorkerId());
        logbrokerWriterLocal.writeSync(feedRequestMessages);
    }

    private UpdateData2Response sendSoapMessage(UUID requestUuid, BsExportIterationContext context,
                                                List<Order> orders) {
        UpdateData2Request.Builder builder = UpdateData2Request.newBuilder()
                .setEngineID(7)
                .setRequestUUID(requestUuid.toString());

        try {
            for (var order : orders) {
                QueryComposer.putOrder(builder, order);
            }
            UpdateData2RequestSoapMessage message = UpdateData2RequestSoapMessage.newBuilder()
                    .setRequest(builder)
                    .setWorkerID((int) context.getWorkerId())
                    .build();
            var soapMessage = new SoapSerializer().serializeRoot(message);

            var response = bannerSystemClient.doRequest(BsUriFactory.UPDATE_DATA_2_SPEC, soapMessage, requestUuid,
                    bsSoapTimeout);
            logger.debug("Response: {}", response);
            SOAPMessage responseMessage = MessageFactory.newInstance().createMessage(new MimeHeaders(),
                    new ByteArrayInputStream(response.getBytes()));
            SoapResponseDeserializer soapResponseDeserializer = new SoapResponseDeserializer(responseMessage);
            return soapResponseDeserializer.deserialize();
        } catch (SOAPException | IOException e) {
            throw new RuntimeException(e);
        }
    }

    private String getParNickName(WorkerSpec workerSpec) {
        return workerSpec.getWorkerType().name().toLowerCase() + "_" + workerSpec.getWorkerId();
    }

    private FeedMessegesLogbrokerWriter getLogbrokerWriter(int shard, WorkerType workerType, long workerId) {
        if (Objects.isNull(logbrokerWriter)) {
            logbrokerWriter = logbrokerWriterFactory.getWriter(shard, workerType, workerId);
        }
        return logbrokerWriter;
    }

    @Override
    public void finish() {
        super.finish();
        if (logbrokerWriter != null) {
            logbrokerWriter.close();
        }
    }
}

