package ru.yandex.direct.jobs.bsexportqueue;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.IntStream;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.GaugeHistogram;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.QueryAndParams.Type;
import ru.yandex.direct.ydb.client.DataQueryResultWrapper;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkState;
import static com.yandex.ydb.table.transaction.TxControl.onlineRo;
import static ru.yandex.direct.jobs.configuration.MaintenanceHelpersYdbConfiguration.MAINTENANCE_HELPERS_YDB_PATH_BEAN;
import static ru.yandex.direct.jobs.configuration.MaintenanceHelpersYdbConfiguration.MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Считаем метрики про очередь экспорта в БК.
 * Работает поверх данных, подготовленных соседней {@link ReplicateToYdbJob}.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0}
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 15),
        needCheck = TestingOnly.class,
        tags = {JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 5, needSchedule = NonDevelopmentEnvironment.class)
@HourglassDaemon(runPeriod = 15)
public class Monitor extends DirectJob {
    static final int[] CAMPAIGN_BINS = {5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 70, 80, 90, 100, 110,
            120, 140, 160, 180, 210, 240, 270, 300, 360, 420, 480, 600, 720, 840, 960, 1200, 1440};
    // клиентов меньше, чем кампаний — поэтому берем только часть корзин CAMPAIGN_BINS
    static final int[] CLIENT_BINS = {10, 20, 30, 45, 60, 90, 120, 180, 240, 360, 480};
    private static final String[] CAMPAIGN_QUEUES = {"std",
            "heavy",
            "nosend",
            "buggy",
            "preprod",
            "internal_ads",
            "fast"
    };
    // интересующих очередей тоже меньше. в остальные, обычно, кампании изредка переносят люди
    private static final String[] CLIENT_QUEUES = {"std",
            "heavy",
            "buggy",
            "internal_ads"
    };
    private static final Labels EXPORT_LABELS = Labels.of("type", "export");
    private static final String CAMPAIGNS_COUNT_SENSOR = "campaigns.count";
    private static final String CLIENTS_COUNT_SENSOR = "clients.count";
    private static final String QUEUE_AGE_SENSOR = "queue.age_minutes";
    private static final String SEQUENCE_AGE_SENSOR = "sequence.age_minutes";
    private static final int SECONDS_IN_MINUTE = 60;
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(3);

    private final YdbClient ydbClient;
    private final ShardHelper shardHelper;
    private final YdbPath ydbPath;
    private final String query;
    private final MetricRegistry exportRegistry;


    @Autowired
    public Monitor(
            ShardHelper shardHelper,
            @Qualifier(MAINTENANCE_HELPERS_YDB_PATH_BEAN) YdbPath ydbPath,
            @Qualifier(MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN) SessionRetryContext sessionRetryContext) {
        this.shardHelper = shardHelper;
        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
        this.ydbPath = ydbPath;
        query = LiveResourceFactory.get("classpath:///monitoring/bsexportqueue.yql").getContent();

        exportRegistry = SolomonUtils.BS_EXPORT_QUEUE_REGISTRY.subRegistry(EXPORT_LABELS);
    }

    private static HistogramCollector campsCollector() {
        return SolomonUtils.explicitHistogram(CAMPAIGN_BINS);
    }

    private static HistogramCollector clientsCollector() {
        return SolomonUtils.explicitHistogram(CLIENT_BINS);
    }

    private static StreamEx<Labels> queueLabelsStream(String[] queues) {
        return StreamEx.of(queues)
                .append("other")
                .map(q -> Labels.of("queue", q))
                .append(Labels.empty());
    }

    private static Labels getQueueLabels(ResultSetReaderWrapped reader) {
        ValueReader parType = reader.getColumn("queue").getOptionalItem();
        if (parType.isOptionalItemPresent()) {
            return Labels.of("queue", parType.getUtf8());
        } else {
            return Labels.empty();
        }
    }

    private static ListValue binsToYdb(int[] bins) {
        ListType type = ListType.of(PrimitiveType.int32());
        var values = IntStream.of(bins)
                .map(m -> m * SECONDS_IN_MINUTE)
                .mapToObj(PrimitiveValue::int32)
                .toArray(Value[]::new);
        return type.newValueOwn(values);
    }

    private static ListValue queuesToYdb(String[] queues) {
        ListType type = ListType.of(PrimitiveType.utf8());
        var values = Arrays.stream(queues)
                .map(PrimitiveValue::utf8)
                .toArray(Value[]::new);
        return type.newValueOwn(values);
    }

    private void putExportCamps(Labels labels, HistogramCollector collector) {
        GaugeHistogram.setGaugeHistogram(exportRegistry, new MetricId(CAMPAIGNS_COUNT_SENSOR, labels), collector);
    }

    private void putExportClients(Labels labels, HistogramCollector collector) {
        GaugeHistogram.setGaugeHistogram(exportRegistry, new MetricId(CLIENTS_COUNT_SENSOR, labels), collector);
    }

    @Override
    public void execute() {
        Params params = Params.copyOf(Map.of("$camp_bins", binsToYdb(CAMPAIGN_BINS),
                "$camp_queues_list", queuesToYdb(CAMPAIGN_QUEUES),
                "$client_bins", binsToYdb(CLIENT_BINS),
                "$client_queues_list", queuesToYdb(CLIENT_QUEUES),
                "$now", PrimitiveValue.datetime(Instant.now()),
                "$secondsInMinute", PrimitiveValue.uint32(SECONDS_IN_MINUTE),
                "$infiniteBin", PrimitiveValue.uint32(Integer.MAX_VALUE)
        ));

        var queryAndParams = new QueryAndParams(ydbPath.getPath(), query, params, Type.READ);
        DataQueryResultWrapper rs = ydbClient.executeQuery(queryAndParams,
                "error retrieving queue stat", true, onlineRo());

        statByAgeBinAndCampaigns(rs.getResultSet(0));
        statByAgeBinAndClients(rs.getResultSet(1));
        maxAge(rs.getResultSet(2), "std");
        maxAge(rs.getResultSet(3), "heavy");
        maxAge(rs.getResultSet(4), "buggy");
        agePercentiles(rs.getResultSet(5), "std");
        agePercentiles(rs.getResultSet(6), "heavy");
        buggyAges(rs.getResultSet(7));
        internalAdsAge(rs.getResultSet(8));
    }

    private void statByAgeBinAndCampaigns(ResultSetReaderWrapped reader) {
        Map<Labels, HistogramCollector> collectors = queueLabelsStream(CAMPAIGN_QUEUES)
                .mapToEntry(key -> campsCollector())
                .toMap();

        while (reader.next()) {
            int bin = reader.getColumn("queue_age_bin").getInt32();
            long count = reader.getColumn("campaigns_count").getUint64();
            Labels labels = getQueueLabels(reader);
            collectors.get(labels).collect(bin, count);
        }
        collectors.forEach(this::putExportCamps);
    }

    private void statByAgeBinAndClients(ResultSetReaderWrapped reader) {
        Map<Labels, HistogramCollector> collectors = queueLabelsStream(CLIENT_QUEUES)
                .mapToEntry(key -> clientsCollector())
                .toMap();

        while (reader.next()) {
            int bin = reader.getColumn("queue_age_max_bin").getInt32();
            long count = reader.getColumn("clients_count").getUint64();
            Labels labels = getQueueLabels(reader);
            collectors.get(labels).collect(bin, count);
        }
        collectors.forEach(this::putExportClients);
    }

    private void maxAge(ResultSetReaderWrapped reader, String queue) {
        Map<Long, Double> ages = new HashMap<>();
        while (reader.next()) {
            long shard = reader.getColumn("shard").getUint32();
            double age = reader.getColumn("queue_age").getFloat64();
            ages.put(shard, age);
        }
        for (long shard : shardHelper.dbShards()) {
            Labels labels = Labels.of("shard", String.valueOf(shard), "queue", queue);
            exportRegistry.gaugeDouble(QUEUE_AGE_SENSOR, labels).set(ages.getOrDefault(shard, 0.0));
        }
    }

    private void agePercentiles(ResultSetReaderWrapped reader, String queue) {
        checkState(reader.next(), "query result is empty");
        for (String p : List.of("p100", "p99", "p95", "p90", "p80")) {
            double age = reader.getColumn(p).getFloat64();
            Labels labels = Labels.of("queue", queue, "percentile", p);
            exportRegistry.gaugeDouble(QUEUE_AGE_SENSOR, labels).set(age);
        }
    }

    private void buggyAges(ResultSetReaderWrapped reader) {
        checkState(reader.next(), "query result is empty");
        Labels labels = Labels.of("queue", "buggy");
        double queueAge = reader.getColumn("queue_age_minutes").getFloat64();
        double sequenceAge = reader.getColumn("sequence_age_minutes").getFloat64();
        exportRegistry.gaugeDouble(QUEUE_AGE_SENSOR, labels).set(queueAge);
        exportRegistry.gaugeDouble(SEQUENCE_AGE_SENSOR, labels).set(sequenceAge);
    }

    private void internalAdsAge(ResultSetReaderWrapped reader) {
        checkState(reader.next(), "query result is empty");
        Labels labels = Labels.of("queue", "internal_ads");
        double age = reader.getColumn("queue_age_minutes").getFloat64();
        exportRegistry.gaugeDouble(QUEUE_AGE_SENSOR, labels).set(age);
    }

    @Override
    public void finish() {
        SolomonUtils.BS_EXPORT_QUEUE_REGISTRY.removeSubRegistry(EXPORT_LABELS);
    }
}
