package ru.yandex.direct.jobs.bsexportqueue;

import java.time.Duration;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.TimeProvider;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба очищает bs_export_queue от поломанных и nosend кампаний
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 17),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}))
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 17),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = NonProductionEnvironment.class)
@Hourglass(periodInSeconds = 5 * 60 * 60, needSchedule = TypicalEnvironment.class)
public class RemoveBrokenCampaignsFromBsExportQueueJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(RemoveBrokenCampaignsFromBsExportQueueJob.class);

    private static final Duration EXPIRE_TIME = Duration.ofHours(36);

    private final BsExportQueueRepository bsExportQueueRepository;
    private final TimeProvider timeProvider;

    @Autowired
    public RemoveBrokenCampaignsFromBsExportQueueJob(BsExportQueueRepository bsExportQueueRepository) {
        this.bsExportQueueRepository = bsExportQueueRepository;
        timeProvider = new TimeProvider();
    }

    /**
     * Конструктор для тестов для указания шарда
     */
    RemoveBrokenCampaignsFromBsExportQueueJob(int shard, BsExportQueueRepository bsExportQueueRepository) {
        super(shard);
        this.bsExportQueueRepository = bsExportQueueRepository;
        timeProvider = new TimeProvider();
    }

    @Override
    public void execute() {
        var campaignIdsToIsLocked = bsExportQueueRepository.getExpiredAndBrokenCampaignIds(getShard(),
                timeProvider.now().minus(EXPIRE_TIME));

        var idsToDelete = EntryStream.of(campaignIdsToIsLocked).filterValues(x -> !x).keys().toList();

        if (!idsToDelete.isEmpty()) {
            logger.info("Got {} campaigns to delete from bs_export_queue and bs_export_specials with ids: {}",
                    idsToDelete.size(), idsToDelete);
            bsExportQueueRepository.delete(getShard(), idsToDelete);
        }

        var idsToWarn = EntryStream.of(campaignIdsToIsLocked).filterValues(Boolean::booleanValue).keys().toList();
        if (!idsToWarn.isEmpty()) {
            logger.warn("Can't delete {} records in bs_export queue with broken campaigns, because they are locked in" +
                    " queue", idsToWarn.size());
            setJugglerStatus(JugglerStatus.WARN, String.format("Can't delete %d records in bs_export queue with " +
                    "broken campaigns, because they are locked in queue", idsToWarn.size()));
        }
    }
}
