package ru.yandex.direct.jobs.bsexportqueue;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.Value;
import one.util.streamex.EntryStream;
import org.jooq.Record;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.dbschema.ppc.enums.BsExportSpecialsParType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.ydbreplication.YdbMapperService;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.QueryAndParams.Type;
import ru.yandex.direct.ydb.client.YdbClient;

import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_SPECIALS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.jobs.configuration.MaintenanceHelpersYdbConfiguration.MAINTENANCE_HELPERS_YDB_PATH_BEAN;
import static ru.yandex.direct.jobs.configuration.MaintenanceHelpersYdbConfiguration.MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Джоба читает данные из ppc.bs_export_queue и загружает их в YDB
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 15),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_0})
@Hourglass(periodInSeconds = 45, needSchedule = NonDevelopmentEnvironment.class)
public class ReplicateToYdbJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(ReplicateToYdbJob.class);
    private static final int MAX_RECORDS_TO_PROCESS = 150_000;
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(3);

    private final DslContextProvider dslContextProvider;
    private final YdbClient ydbClient;
    private final YdbPath ydbPath;
    private final YdbMapperService ydbMapperService;

    @Autowired
    public ReplicateToYdbJob(
            DslContextProvider dslContextProvider,
            @Qualifier(MAINTENANCE_HELPERS_YDB_PATH_BEAN) YdbPath ydbPath,
            @Qualifier(MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN) SessionRetryContext sessionRetryContext,
            YdbMapperService ydbMapperService) {
        this.dslContextProvider = dslContextProvider;
        this.ydbMapperService = ydbMapperService;
        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
        this.ydbPath = ydbPath;
    }

    @Override
    public void execute() {
        int shard = getShard();

        logger.info("start fetching");
        List<StructValue> values = dslContextProvider.ppc(shard)
                .select(BS_EXPORT_QUEUE.CID, BS_EXPORT_QUEUE.CAMPS_NUM, BS_EXPORT_QUEUE.BANNERS_NUM,
                        BS_EXPORT_QUEUE.CONTEXTS_NUM, BS_EXPORT_QUEUE.BIDS_NUM, BS_EXPORT_QUEUE.PRICES_NUM,
                        BS_EXPORT_QUEUE.IS_FULL_EXPORT, BS_EXPORT_QUEUE.SEQ_TIME, BS_EXPORT_QUEUE.QUEUE_TIME,
                        BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME, BS_EXPORT_QUEUE.PAR_ID,
                        CAMPAIGNS.CLIENT_ID, CAMPAIGNS.AGENCY_ID, CAMPAIGNS.TYPE,
                        BS_EXPORT_SPECIALS.PAR_TYPE)
                .from(BS_EXPORT_QUEUE)
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BS_EXPORT_QUEUE.CID))
                .leftJoin(BS_EXPORT_SPECIALS).on(BS_EXPORT_SPECIALS.CID.eq(BS_EXPORT_QUEUE.CID))
                .limit(MAX_RECORDS_TO_PROCESS)
                .fetch(this::mapper);
        logger.info("fetched {} records from MySQL", values.size());

        logger.info("start ydb");
        if (values.isEmpty()) {
            deleteFromYdbBsExportQueue(shard);
        } else {
            replaceIntoYdbBsExportQueue(shard, values);
        }
        logger.info("done ydb");

        if (values.size() == MAX_RECORDS_TO_PROCESS) {
            String message = String.format("Reached maximum number of rows in bs_export_queue for export to Ydb: " +
                    "limit %d, shard %d", MAX_RECORDS_TO_PROCESS, shard);
            logger.warn(message);
            setJugglerStatus(JugglerStatus.WARN, message);
        }
    }

    private void replaceIntoYdbBsExportQueue(int shard, List<StructValue> values) {
        String query = "DECLARE $shard AS Uint32;\n" +
                "DECLARE $values AS List<Struct<shard:Uint32,cid:Uint64," +
                "camps_num:Uint64,banners_num:Uint64,contexts_num:Uint64,bids_num:Uint64,prices_num:Uint64," +
                "is_full_export:Bool,seq_time:Datetime,queue_time:Datetime,full_export_seq_time:Datetime," +
                "par_id:Uint32?,ClientID:Uint64?,AgencyID:Uint64?,type:Utf8?,par_type:Utf8?>>;\n" +

                "DELETE FROM `bs_export_queue` WHERE shard = $shard" +
                " AND cid NOT IN ListMap($values, ($v) -> { RETURN $v.cid });\n" +
                "REPLACE INTO `bs_export_queue` SELECT * FROM AS_TABLE($values);";

        Params params = Params.of("$shard", PrimitiveValue.uint32(shard),
                "$values", ListValue.of(values.toArray(StructValue[]::new)));

        var queryAndParams = new QueryAndParams(ydbPath.getPath(), query, params, Type.WRITE);
        ydbClient.executeQuery(queryAndParams, "error replacing into bs_export_queue", true);
    }

    private void deleteFromYdbBsExportQueue(int shard) {
        String query = "DECLARE $shard AS Uint32;\n" +
                "DELETE FROM `bs_export_queue` WHERE shard = $shard;";
        Params params = Params.of("$shard", PrimitiveValue.uint32(shard));

        var queryAndParams = new QueryAndParams(ydbPath.getPath(), query, params, Type.WRITE);
        ydbClient.executeQuery(queryAndParams, "error deleting from bs_export_queue", true);
    }

    private StructValue mapper(Record record) {
        Map<String, Value> ydbValueByFieldName = EntryStream.of(record.intoMap())
                .mapToValue(this::convertToYdb)
                .append("shard", PrimitiveValue.uint32(getShard()))
                .toMap();
        return StructValue.of(ydbValueByFieldName);
    }

    private Value convertToYdb(String fieldName, Object value) {
        switch (fieldName) {
            case "cid":
            case "camps_num":
            case "banners_num":
            case "contexts_num":
            case "bids_num":
            case "prices_num":
                return PrimitiveValue.uint64((Long) value);

            case "ClientID":
            case "AgencyID":
                return ydbMapperService.convertToOptionalUint64((Long) value);

            case "is_full_export":
                Boolean booleanValue = RepositoryUtils.booleanFromLong((Long) value);
                return PrimitiveValue.bool(booleanValue);

            case "seq_time":
            case "queue_time":
            case "full_export_seq_time":
                return ydbMapperService.convertToLocalDateTime(value);

            case "par_id":
                return ydbMapperService.convertToOptionalUint32((Long) value);

            case "par_type":
                return ydbMapperService.convertToOptionalUtf8((BsExportSpecialsParType) value);

            case "type":
                return ydbMapperService.convertToOptionalUtf8((CampaignsType) value);

            default:
                throw new UnsupportedOperationException("no mapping for key " + fieldName);
        }
    }
}
