package ru.yandex.direct.jobs.calltracking

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyName
import ru.yandex.direct.common.db.PpcPropertyType
import ru.yandex.direct.core.entity.calltrackingsettings.repository.CalltrackingSettingsRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YqlRowMapper
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.misc.io.ClassPathResourceInputStreamSource
import java.time.LocalDateTime
import javax.annotation.ParametersAreNonnullByDefault

const val LAST_TIME_PROPERTY = "sync_calltracking_settings_counter_statuses_job_last_time"

/**
 *  Джоба проставляет флаг {@code calltracking_settings.is_counter_available = 0}
 *  для счетчиков которые числяться в экспорте метрики [//home/metrika/export/counters] как удаленные
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 3, minutes = 5),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_CALLTRACKING],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT])
    ]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly::class)
@ParametersAreNonnullByDefault
class SyncCalltrackingSettingsCounterStatusesJob @Autowired constructor(
    private val ytProvider: YtProvider,
    private val calltrackingSettingsRepository: CalltrackingSettingsRepository,
    private val shardHelper: ShardHelper,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : DirectJob() {
    private val ytClusters = listOf(YtCluster.HAHN, YtCluster.ARNOLD)

    companion object {
        private val logger = LoggerFactory.getLogger(SyncCalltrackingSettingsCounterStatusesJob::class.java)
    }

    override fun execute() {
        val now = LocalDateTime.now()

        // Необходимо, чтобы джоба успешно выполнялась раз в сутки. Запускаем ее каждый час
        // и записываем время последнего запуска в свойство [PpcPropertyNames.LAST_TIME_PROPERTY]
        // Каждый час при запуске проверяем, прошли ли с момента успешного запуска одни сутки
        val lastTimePropertyName = PpcPropertyName(LAST_TIME_PROPERTY, PpcPropertyType.LOCAL_DATE_TIME)
        val lastTimeProperty = ppcPropertiesSupport.get(lastTimePropertyName)

        if (!isTimeToExecute(lastTimeProperty, now)) {
            logger.info("Skip processing. Too early to execute job")
            return
        }

        logger.info("Start sync statuses for counter used in calltracking_settings")

        val deletedCounterIds = getDeletedCounters()
        logger.info("Found {} deleted counters", deletedCounterIds.size)
        if (deletedCounterIds.isNotEmpty()) {
            // обновляем все настройки счетчики которых в выгрузке были помечены как удаленные
            shardHelper.dbShards().forEach {
                logger.info("Update calltracking_settings counter availability for shard {}", it)
                calltrackingSettingsRepository.updateCounterAvailability(it, deletedCounterIds, false)
            }
        }

        lastTimeProperty.set(now)
        logger.info("Finished sync statuses for counter used in calltracking_settings")
    }

    private fun getDeletedCounters(): List<Long> {
        return YtExecutionUtil.executeWithFallback(
            ytClusters,
            ytProvider::getOperator,
            this::getDeletedCounters)
    }

    private fun getDeletedCounters(ytOperator: YtOperator): List<Long> {
        val deletedCountersQuery = ClassPathResourceInputStreamSource("calltracking/CollectDeletedCounters.yql")
            .readLines().joinToString("\n")
        val mapper = YqlRowMapper {
            it.getLong("counterId")
        }

        return ytOperator.yqlQuery(deletedCountersQuery, mapper)
    }

    private fun isTimeToExecute(lastTimeProperty: PpcProperty<LocalDateTime>?, now: LocalDateTime?): Boolean {
        val lastTime = lastTimeProperty?.get() ?: return true
        return lastTime.plusDays(1).isBefore(now)
    }
}
