package ru.yandex.direct.jobs.calltracking;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettingsWithCampaignId;
import ru.yandex.direct.core.entity.calltracking.model.CampCalltrackingPhones;
import ru.yandex.direct.core.entity.calltracking.model.SettingsPhone;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.CalltrackingSettingsRepository;
import ru.yandex.direct.core.entity.campcalltrackingphones.service.CampCalltrackingPhonesService;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба синхронизирует номера для коллтрекинга на сайте.
 * Если в calltracking_settings в списке номеров есть какой-то номер, но его нет в client_phones,
 * то заводим такой номер и привязываем его к подменнику Телефонии.
 * Если номер для коллтрекинга на сайте есть в client_phones, но его нет ни в одной записи calltracking_settings,
 * то помечаем этот номер удалённым (is_deleted = true).
 * Если есть запись и в client_phones, и в calltracking_settings, но счётчики Метрики различаются,
 * то обновляем счётчик у номера в client_phones и перепривязываем его в Телефонии.
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.DIRECT_CALLTRACKING},
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING},
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 10 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class SyncCalltrackingTelephonyPhonesJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(SyncCalltrackingTelephonyPhonesJob.class);

    private final CalltrackingSettingsRepository calltrackingSettingsRepository;
    private final CampCalltrackingPhonesService campCalltrackingPhonesService;
    private final ClientPhoneService clientPhoneService;
    private final TelephonyPhoneService telephonyPhoneService;

    @Autowired
    public SyncCalltrackingTelephonyPhonesJob(CalltrackingSettingsRepository calltrackingSettingsRepository,
                                              CampCalltrackingPhonesService campCalltrackingPhonesService,
                                              ClientPhoneService clientPhoneService,
                                              TelephonyPhoneService telephonyPhoneService) {
        this.calltrackingSettingsRepository = calltrackingSettingsRepository;
        this.campCalltrackingPhonesService = campCalltrackingPhonesService;
        this.clientPhoneService = clientPhoneService;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    public SyncCalltrackingTelephonyPhonesJob(int shard,
                                              CalltrackingSettingsRepository calltrackingSettingsRepository,
                                              CampCalltrackingPhonesService campCalltrackingPhonesService,
                                              ClientPhoneService clientPhoneService,
                                              TelephonyPhoneService telephonyPhoneService) {
        super(shard);
        this.calltrackingSettingsRepository = calltrackingSettingsRepository;
        this.campCalltrackingPhonesService = campCalltrackingPhonesService;
        this.clientPhoneService = clientPhoneService;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    @Override
    public void execute() {
        int shard = getShard();
        Map<ClientId, Map<Long, ClientPhone>> clientPhonesByIdByClientId =
                clientPhoneService.getSiteTelephonyByIdAndClientId(shard);
        Map<ClientId, List<CalltrackingSettingsWithCampaignId>> calltrackingSettingsWithCampaignIdByClientId =
                calltrackingSettingsRepository.getWithCampaignId(shard);
        Map<Long, List<Long>> clientPhoneIdsByCampaignId =
                campCalltrackingPhonesService.getClientPhoneIdsByCampaignId(shard);
        var clientIds =
                StreamEx.ofKeys(calltrackingSettingsWithCampaignIdByClientId)
                        .append(clientPhonesByIdByClientId.keySet())
                        .distinct()
                        .toList();
        boolean isTelephonyAllocatesSuccess = true;
        for (var clientId : clientIds) {
            List<CalltrackingSettingsWithCampaignId> calltrackingSettingsWithCampaignId =
                    calltrackingSettingsWithCampaignIdByClientId.getOrDefault(clientId, emptyList());
            try {
                isTelephonyAllocatesSuccess &= syncCalltrackingTelephonyPhonesForClientId(
                        shard,
                        clientId,
                        calltrackingSettingsWithCampaignId,
                        clientPhonesByIdByClientId,
                        clientPhoneIdsByCampaignId
                );
            } catch (RuntimeException ex) {
                logger.error("Couldn't synchronize phones for calltracking on site for clientId {}",
                        clientId.asLong(), ex);
            }
        }
        if (!isTelephonyAllocatesSuccess) {
            setJugglerStatus(JugglerStatus.CRIT, "Failed to allocate at least one phone in Telephony");
        }
    }

    /**
     * @return {@code false}, если не удалось выделить хотя бы один номер Телефонии, если это требовалось
     * Иначе {@code true}
     */
    private boolean syncCalltrackingTelephonyPhonesForClientId(
            int shard,
            ClientId clientId,
            List<CalltrackingSettingsWithCampaignId> calltrackingSettingsWithCampaignId,
            Map<ClientId, Map<Long, ClientPhone>> clientPhonesByIdByClientId,
            Map<Long, List<Long>> clientPhoneIdsByCampaignId) {
        Map<Long, ClientPhone> clientPhonesById = clientPhonesByIdByClientId.getOrDefault(clientId, emptyMap());
        PhonesToUpdateContainer container = getPhonesToUpdate(
                calltrackingSettingsWithCampaignId, clientPhonesById, clientPhoneIdsByCampaignId);

        if (container.newPhonesWithCampaignId.isEmpty() &&
                container.clientPhonesWithNewCounterId.isEmpty() &&
                container.clientPhonesToDelete.isEmpty()) {
            // no changes
            return true;
        }
        logChangesSafely(clientId, container,
                calltrackingSettingsWithCampaignId, clientPhoneIdsByCampaignId, clientPhonesById);

        // если записи нет, то нужно выдать номер в Телефонии, и записать в client_phones
        if (!container.newPhonesWithCampaignId.isEmpty()) {
            List<Pair<ClientPhone, Long>> phonesToAdd = container.newPhonesWithCampaignId.stream()
                    .map(newPhoneWithCampaignId -> {
                        ClientPhone clientPhone = new ClientPhone()
                                .withClientId(clientId)
                                .withPhoneType(ClientPhoneType.TELEPHONY)
                                .withPhoneNumber(new PhoneNumber().withPhone(newPhoneWithCampaignId.getKey()))
                                .withCounterId(newPhoneWithCampaignId.getValue().getCounterId())
                                .withPermalinkId(0L)
                                .withIsDeleted(false);
                        return Pair.of(clientPhone, newPhoneWithCampaignId.getValue().getCampaignId());
                    })
                    .collect(Collectors.toList());
            List<CampCalltrackingPhones> campCalltrackingPhones =
                    clientPhoneService.addNewCalltrackingOnSitePhones(clientId, phonesToAdd);

            if (campCalltrackingPhones.size() < phonesToAdd.size()) {
                return false;
            }
            // Запишем добавленные номера, чтобы создать для них записи в camp_calltracking_phones
            container.campCalltrackingPhones.addAll(campCalltrackingPhones);
        }

        // если запись есть, но номер счетчика не совпадает, должны обновить его в Телефонии
        // и обновить счетчик в client_phones
        if (!container.clientPhonesWithNewCounterId.isEmpty()) {
            telephonyPhoneService.updateCounterIds(shard, clientId, container.clientPhonesWithNewCounterId);
        }

        // Записи, которые не были привязаны ни к одной настройке колтрекинга, помечаются удалёнными
        if (!container.clientPhonesToDelete.isEmpty()) {
            telephonyPhoneService.setCalltrackingOnSitePhonesDeleted(
                    shard, clientId, container.clientPhonesToDelete.values());
        }

        // Заполняем таблицу camp_calltracking_phones
        if (!container.newPhonesWithCampaignId.isEmpty() || !container.clientPhonesToDelete.isEmpty()) {
            Set<Long> clientPhoneIdsToDelete = listToSet(
                    container.campCalltrackingPhones, CampCalltrackingPhones::getClientPhoneId);
            clientPhoneIdsToDelete.addAll(container.clientPhonesToDelete.keySet());
            campCalltrackingPhonesService.updateCampCalltrackingPhonesList(
                    shard, clientPhoneIdsToDelete, container.campCalltrackingPhones);
        }
        return true;
    }

    private void logChangesSafely(
            ClientId clientId,
            PhonesToUpdateContainer container,
            @Nullable List<CalltrackingSettingsWithCampaignId> calltrackingSettingsWithCampaignId,
            @Nullable Map<Long, List<Long>> clientPhoneIdsByCampaignId,
            @Nullable Map<Long, ClientPhone> clientPhonesById) {
        try {
            logChanges(clientId, container,
                    calltrackingSettingsWithCampaignId, clientPhoneIdsByCampaignId, clientPhonesById);
        } catch (RuntimeException e) {
            String messageWithMethodArgs = "Can't log changes by params clientId: " + clientId + ", " +
                    "container: " + container + ", " +
                    "calltrackingSettingsWithCampaignId: " + calltrackingSettingsWithCampaignId + ", " +
                    "clientPhoneIdsByCampaignId: " + clientPhoneIdsByCampaignId + ", " +
                    "clientPhonesById: " + clientPhonesById;
            logger.warn(messageWithMethodArgs, e);
        }
    }

    private void logChanges(
            ClientId clientId,
            PhonesToUpdateContainer container,
            @Nullable List<CalltrackingSettingsWithCampaignId> calltrackingSettingsWithCampaignId,
            @Nullable Map<Long, List<Long>> clientPhoneIdsByCampaignId,
            @Nullable Map<Long, ClientPhone> clientPhonesById) {
        String phonesToDeleteString =
                container.clientPhonesToDelete.values().stream()
                        .map(phone -> "{id: " + phone.getId() +
                                ", phone: " + getPhoneOrNull(phone) + "}")
                        .collect(Collectors.joining(", ", "[", "]"));
        String newPhonesToTrack =
                container.newPhonesWithCampaignId.stream()
                        .map(phoneWithCamp -> "{phone: " + phoneWithCamp.getLeft() +
                                ", settingsId: " + phoneWithCamp.getRight().getCalltrackingSettingsId() +
                                ", campaignId: " + phoneWithCamp.getRight().getCampaignId() + "}")
                        .collect(Collectors.joining(", ", "[", "]"));
        String trackingPhonesWithNewCounter =
                container.clientPhonesWithNewCounterId.stream()
                        .map(phoneWithCounter -> "{phoneId: " + phoneWithCounter.getLeft().getId() +
                                ", phone: " + getPhoneOrNull(phoneWithCounter.getLeft()) +
                                ", counterId: " + phoneWithCounter.getRight() + "}")
                        .collect(Collectors.joining(", ", "[", "]"));

        Map<Long, List<Long>> campaignIdsByPhoneId;
        if (clientPhoneIdsByCampaignId != null) {
            campaignIdsByPhoneId = EntryStream.of(clientPhoneIdsByCampaignId)
                    .flatMapValues(Collection::stream)
                    .invert()
                    .grouping();
        } else {
            campaignIdsByPhoneId = emptyMap();
        }

        String existingClientPhones =
                clientPhonesById == null ? null : clientPhonesById.values().stream()
                        .map(phone -> "{id: " + phone.getId() +
                                ", phone: " + getPhoneOrNull(phone) +
                                ", campaignIds: " + campaignIdsByPhoneId.get(phone.getId()) +
                                "}")
                        .collect(Collectors.joining(", ", "[", "]"));

        String actualSettings =
                calltrackingSettingsWithCampaignId == null
                        ? null
                        : calltrackingSettingsWithCampaignId.stream()
                        .map(settings -> "{campaignId: " + settings.getCampaignId() +
                                ", settingsId: " + settings.getCalltrackingSettingsId() +
                                ", phonesToTrack: " + settings.getPhonesToTrack() +
                                "}")
                        .collect(Collectors.joining(", ", "[", "]"));
        logger.info("Changes are required for client " + clientId +
                "\ntracking phones to delete: " + phonesToDeleteString +
                "\nnew phones to track: " + newPhonesToTrack +
                "\ncounter changed: " + trackingPhonesWithNewCounter +
                "\n\nCurrent phones: " + existingClientPhones +
                "\nCurrent settings: " + actualSettings
        );
    }

    private String getPhoneOrNull(ClientPhone phone) {
        return Optional.ofNullable(phone.getTelephonyPhone()).map(PhoneNumber::getPhone).orElse(null);
    }

    private PhonesToUpdateContainer getPhonesToUpdate(
            List<CalltrackingSettingsWithCampaignId> calltrackingSettingsWithCampaignId,
            Map<Long, ClientPhone> clientPhonesById,
            Map<Long, List<Long>> clientPhoneIdsByCampaignId) {
        List<Pair<String, CalltrackingSettingsWithCampaignId>> newPhonesWithSettings = new ArrayList<>();
        List<Pair<ClientPhone, Long>> clientPhonesWithNewCounterId = new ArrayList<>();
        List<CampCalltrackingPhones> campCalltrackingPhones = new ArrayList<>();
        Map<Long, ClientPhone> clientPhonesToDelete = new HashMap<>(clientPhonesById);
        for (var settingsWithCampaignId : calltrackingSettingsWithCampaignId) {
            long campaignId = settingsWithCampaignId.getCampaignId();
            List<String> phonesToTrack = mapList(settingsWithCampaignId.getPhonesToTrack(), SettingsPhone::getPhone);
            List<Long> clientPhoneIds = clientPhoneIdsByCampaignId.getOrDefault(campaignId, emptyList());

            if (!settingsWithCampaignId.getIsAvailableCounter()) {
                continue;
            }

            for (String phone : phonesToTrack) {
                Optional<ClientPhone> clientPhoneOptional =
                        findClientPhoneByPhoneAndPhoneIds(phone, clientPhoneIds, clientPhonesById);
                if (clientPhoneOptional.isPresent()) {
                    // Существующий клиентский номер
                    ClientPhone clientPhone = clientPhoneOptional.get();
                    clientPhonesToDelete.remove(clientPhone.getId());
                    campCalltrackingPhones.add(new CampCalltrackingPhones()
                            .withClientPhoneId(clientPhone.getId())
                            .withCid(campaignId)
                    );
                    if (clientPhone.getTelephonyPhone() != null &&
                            !clientPhone.getCounterId().equals(settingsWithCampaignId.getCounterId())) {
                        // Если у номера есть подменник, но счётчик Метрики в настройках указан другой,
                        // то обновляем счётчик Метрики у номера
                        clientPhonesWithNewCounterId.add(
                                Pair.of(clientPhone, settingsWithCampaignId.getCounterId()));
                    }
                } else {
                    // Не нашли клиентский номер - добавляем его
                    newPhonesWithSettings.add(Pair.of(phone, settingsWithCampaignId));
                }
            }
        }
        return new PhonesToUpdateContainer(newPhonesWithSettings, clientPhonesWithNewCounterId,
                campCalltrackingPhones, clientPhonesToDelete);
    }

    private Optional<ClientPhone> findClientPhoneByPhoneAndPhoneIds(String phone,
                                                                    List<Long> clientPhoneIds,
                                                                    Map<Long, ClientPhone> clientPhonesById) {
        return clientPhonesById.entrySet()
                .stream()
                .filter(e -> clientPhoneIds.contains(e.getKey()))
                .filter(e -> phone.equals(e.getValue().getPhoneNumber().getPhone()))
                .map(Map.Entry::getValue)
                .findFirst();
    }

    private static class PhonesToUpdateContainer {
        private final List<Pair<String, CalltrackingSettingsWithCampaignId>> newPhonesWithCampaignId;
        private final List<Pair<ClientPhone, Long>> clientPhonesWithNewCounterId;
        private final List<CampCalltrackingPhones> campCalltrackingPhones;
        private final Map<Long, ClientPhone> clientPhonesToDelete;

        PhonesToUpdateContainer(List<Pair<String, CalltrackingSettingsWithCampaignId>> newPhonesWithCampaignId,
                                List<Pair<ClientPhone, Long>> clientPhonesWithNewCounterId,
                                List<CampCalltrackingPhones> campCalltrackingPhones,
                                Map<Long, ClientPhone> clientPhonesToDelete) {
            this.newPhonesWithCampaignId = newPhonesWithCampaignId;
            this.clientPhonesWithNewCounterId = clientPhonesWithNewCounterId;
            this.campCalltrackingPhones = campCalltrackingPhones;
            this.clientPhonesToDelete = clientPhonesToDelete;
        }

        @Override
        public String toString() {
            return "PhonesToUpdateContainer{" +
                    "newPhonesWithCampaignId=" + newPhonesWithCampaignId +
                    ", clientPhonesWithNewCounterId=" + clientPhonesWithNewCounterId +
                    ", campCalltrackingPhones=" + campCalltrackingPhones +
                    ", clientPhonesToDelete=" + clientPhonesToDelete +
                    '}';
        }
    }
}
