package ru.yandex.direct.jobs.campaign;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.service.CpmPriceCampaignService;
import ru.yandex.direct.core.entity.strategy.type.withinventoridata.StrategyInventoriDataRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceApprover;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceStatusApprove;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.inventori.InventoriCampaignsPredictionClient;
import ru.yandex.direct.inventori.InventoriGeneralCampaignsPredictionResponse;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsCpmPrice.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.inventori.InventoriGeneralCampaignsPredictionResponse.TRAFFIC_LIGHT_GREEN;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.LOGIN_OVAZHNEV},
                status = {JugglerStatus.OK, JugglerStatus.CRIT},
                method = NotificationMethod.EMAIL
        ),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM})
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ApproveCpmPriceCampaignFlightStatusApproveJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ApproveCpmPriceCampaignFlightStatusApproveJob.class);
    private final DslContextProvider dslContextProvider;
    private final CpmPriceCampaignService cpmPriceCampaignService;
    private final InventoriCampaignsPredictionClient inventoriClient;
    private final StrategyInventoriDataRepository strategyInventoriDataRepository;

    @Autowired
    public ApproveCpmPriceCampaignFlightStatusApproveJob(
            DslContextProvider dslContextProvider,
            CpmPriceCampaignService cpmPriceCampaignService,
            InventoriCampaignsPredictionClient inventoriClient,
            StrategyInventoriDataRepository strategyInventoriDataRepository) {
        this.dslContextProvider = dslContextProvider;
        this.cpmPriceCampaignService = cpmPriceCampaignService;
        this.inventoriClient = inventoriClient;
        this.strategyInventoriDataRepository = strategyInventoriDataRepository;
    }

    @Override
    public void execute() {
        Map<Integer, List<CpmPriceCampaign>> cpmPriceCampaignsWaitingAutoApproveByShard =
                cpmPriceCampaignService.getCpmPriceCampaignsWaitingAutoApproveByShard();
        String problemCampaignMessage = null;
        for (var entry : cpmPriceCampaignsWaitingAutoApproveByShard.entrySet()) {
            Integer shard = entry.getKey();
            List<CpmPriceCampaign> campaigns = entry.getValue();
            Set<Long> greenCampaigns = new HashSet<>();
            Set<Long> redCampaigns = new HashSet<>();
            Map<Long, Long> campaignAuctionProbabilityByStrategyId = new HashMap<>();
            boolean interrupted = false;
            try {
                for (var campaign : campaigns) {
                    Long cid = campaign.getId();
                    InventoriGeneralCampaignsPredictionResponse campaignPrediction;
                    try {
                        campaignPrediction = inventoriClient.generalCampaignsPrediction(cid);
                        if (campaignPrediction.getTrafficLightColour().equals(TRAFFIC_LIGHT_GREEN)) {
                            long auctionProbability =  0L;
                            long forecastImpression = nvl(campaignPrediction.getForecastImpressions(), 0L);
                            long impressionCapacity = nvl(campaignPrediction.getImpressionsCapacity(), 0L);
                            if(impressionCapacity != 0 && forecastImpression != 0) {
                                auctionProbability = 1000000 * forecastImpression / impressionCapacity;
                                campaignAuctionProbabilityByStrategyId.put(campaign.getStrategyId(), auctionProbability);
                            }
                            greenCampaigns.add(cid);
                        } else {
                            redCampaigns.add(cid);
                        }
                    } catch (InterruptedException e) {
                        interrupted = true;
                        break;
                    } catch (RuntimeException e) {
                        String shortMessage = String.format("Inventori request failed for shard %d cid %d" +
                                " message %s", shard, cid, e.getMessage());
                        logger.error(shortMessage + " cause: " + e.getCause());
                        if (problemCampaignMessage == null) {
                            problemCampaignMessage = shortMessage;
                        }
                    }
                }
            } finally {
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
            }
            if (interrupted) {
                break;
            }
            logger.info("In shard {} traffic light green campaigns {}", shard, greenCampaigns);
            logger.info("In shard {} traffic light red campaigns {}", shard, redCampaigns);
            approveCampaigns(shard, greenCampaigns);
            saveStrategyData(shard, campaignAuctionProbabilityByStrategyId);
        }
        if (problemCampaignMessage != null) {
            setJugglerStatus(JugglerStatus.CRIT, problemCampaignMessage);
        }
    }

    private void approveCampaigns(int shard, Collection<Long> campaignIds) {
        if (campaignIds.size() == 0) {
            return;
        }

        int rows = dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_CPM_PRICE.join(CAMPAIGNS).on(CAMPAIGNS_CPM_PRICE.CID.eq(CAMPAIGNS.CID)))
                .set(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE, CampaignsCpmPriceStatusApprove.Yes)
                .set(CAMPAIGNS_CPM_PRICE.APPROVER, CampaignsCpmPriceApprover.inventori)
                .set(CAMPAIGNS_CPM_PRICE.APPROVE_DATE, LocalDateTime.now())
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .where(CAMPAIGNS_CPM_PRICE.CID.in(campaignIds))
                .execute();

        if (rows > 0) {
            logger.info("In shard {} approved {} of {} campaigns: {} ", shard, rows, campaignIds.size(), campaignIds);
            logger.info(String.format("CampaignsCpmPrice: " +
                            "{\"campaignIds\": [%s], \"statusApprove\": \"%s\", \"approver\": \"%s\"}",
                    String.join(",", mapList(campaignIds, Object::toString)),
                    CampaignsCpmPriceStatusApprove.Yes.getLiteral(),
                    CampaignsCpmPriceApprover.inventori.getLiteral()));
        }
    }
    private void saveStrategyData(int shard, Map<Long, Long> campaignAuctionProbabilityByStrategyId) {
        if (campaignAuctionProbabilityByStrategyId.isEmpty()) {
            return;
        }
        strategyInventoriDataRepository.save(shard, campaignAuctionProbabilityByStrategyId);
        logger.info(String.format("CampaignsCpmPrice: strategyId:auctionProbability -> %s",
                campaignAuctionProbabilityByStrategyId.entrySet().stream()
                        .map(e -> e.getKey() + ":" + e.getValue()).collect(Collectors.joining(","))));
    }
}
