package ru.yandex.direct.jobs.campaign;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyFinishedByDate;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.CampFinishedMailNotification;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.Collections.unmodifiableList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Рассылает предупреждения об остановившихся по дате окончания кампаниям клиентам и менеджерам.
 * Рассылка ведётся о кампаниях, для которых наступила установленная в них дата окончания, но
 * на кампании ещё остались деньги. Работает с текстовыми и медийными кампаниями.
 * Для универсальных кампаний нотификации не отправляются
 * <p>
 * Предупреждение отправляется сразу после наступления даты окончания, через 3 дня и через 7 дней
 * после остановки кампании. Текст предупреждения одинаковый.
 * <p>
 * Кампания останавливается в БК ровно в полночь даты окончания по Московскому времени. Запускать
 * можно в любое время суток (учитываются только даты), но логичнее это делать ближе ко
 * времени остановки кампаний, т.е. к полуночи.
 * <p>
 * Факт отправки предупреждения нигде не фиксируется,
 * а необходимость его отправки определяется по наличию денег на кампании и числу прошедших с даты окончания средств.
 * <p>
 * Если в какой-то день джоб не будет запущен или его выполнение будет по какой-то причине прервано,
 * часть пользователей не получат предупреждений.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4),
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 7 0 * * ?", needSchedule = TypicalEnvironment.class)
public class CampFinishedByDateWarningSender extends DirectShardedJob {

    /**
     * Сразу после наступления даты окончания, через 3 дня и через 7 дней
     */
    static final List<Integer> WARN_WHEN_FINISHED_AFTER_DAYS = unmodifiableList(Arrays.asList(0, 3, 7));
    private static final Logger logger = LoggerFactory.getLogger(CampFinishedByDateWarningSender.class);

    private final CampaignService campaignService;
    private final EventLogService eventLogService;
    private final NotificationService notificationService;
    private final UserService userService;

    @Autowired
    public CampFinishedByDateWarningSender(CampaignService campaignService, EventLogService eventLogService,
                                           NotificationService notificationService, UserService userService) {
        this.campaignService = campaignService;
        this.eventLogService = eventLogService;
        this.notificationService = notificationService;
        this.userService = userService;
    }

    CampFinishedByDateWarningSender(int shard, CampaignService campaignService, EventLogService eventLogService,
                                    NotificationService notificationService, UserService userService) {
        super(shard);
        this.campaignService = campaignService;
        this.eventLogService = eventLogService;
        this.notificationService = notificationService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        final List<CampaignForNotifyFinishedByDate> campaigns = campaignService.getCampaignsForNotifyFinishedByDate(
                getShard(), WARN_WHEN_FINISHED_AFTER_DAYS);
        final Set<Long> userIds = campaigns.stream().map(CampaignForNotifyFinishedByDate::getUserId).collect(toSet());

        final Map<Long, User> users = userService.massGetUser(userIds).stream().collect(toMap(User::getId, identity()));

        final List<CampFinishedMailNotification> notifications = campaigns.stream()
                .map(campaign -> getCampFinishedMailNotification(users.get(campaign.getUserId()), campaign))
                .collect(toList());

        if (logger.isTraceEnabled()) {
            logger.trace("Finished by date campaign IDs: {}", notifications.stream()
                    .map(CampFinishedMailNotification::getCampaignId)
                    .map(Object::toString)
                    .collect(joining(",")));
        }
        for (CampFinishedMailNotification n : notifications) {
            logger.info("Adding notification for campaign with cid = {}", n.getCampaignId());
            try {
                eventLogService.addCampFinishedEventLog(n.getCampaignId(), n.getFinishDate(), n.getClientId());
                notificationService.addNotification(n);
            } catch (RuntimeException e) {
                logger.error("Notification failed for campaign with cid = " + n.getCampaignId(), e);
            }
        }
    }

    private CampFinishedMailNotification getCampFinishedMailNotification(User user,
                                                                         CampaignForNotifyFinishedByDate campaign) {
        return new CampFinishedMailNotification()
                .withCampaignId(campaign.getId())
                .withCampaignName(campaign.getName())
                .withAgencyUid(campaign.getAgencyUserId())
                .withClientId(campaign.getClientId())
                .withClientUserId(campaign.getUserId())
                .withClientLogin(user.getLogin())
                .withClientPhone(user.getPhone())
                .withClientEmail(isNotBlank(campaign.getEmail()) ? campaign.getEmail() : user.getEmail())
                .withClientFullName(user.getFio())
                .withClientLang(user.getLang().getLangString())
                .withFinishDate(campaign.getFinishTime());
    }
}
