package ru.yandex.direct.jobs.campaign

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.spring.TestingComponent
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.NonProductionEnvironment
import ru.yandex.direct.jobs.campaign.repository.CleanTestingCampaignsRepository
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.Duration

@TestingComponent
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 2, hours = 4),
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DARKKEKS],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT],
    )],
    needCheck = NonProductionEnvironment::class,
)
@Hourglass(cronExpression = "0 15 1 * * ?", needSchedule = NonProductionEnvironment::class)
class CleanTestingCampaignsJob(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val shardHelper: ShardHelper,
    private val cleanTestingCampaignsRepository: CleanTestingCampaignsRepository,
    private val rbacService: RbacService,
    private val campaignService: CampaignService,
    private val campaignRepository: CampaignRepository,
    private val grutApiService: GrutApiService,
) : DirectShardedJob() {

    private val logger = LoggerFactory.getLogger(CleanTestingCampaignsJob::class.java)

    private val deletionAge: Duration
        get() = ppcPropertiesSupport.get(PpcPropertyNames.CLEAN_TESTING_CAMPAIGNS_AGE_MINUTES).get()
            ?.let { Duration.ofMinutes(it) }
            ?: Duration.ofDays(30)

    override fun execute() {
        if (Environment.getCached().isProduction) {
            throw IllegalStateException("Job must not be launched in production")
        }

        val logins: Set<String> = ppcPropertiesSupport
            .get(PpcPropertyNames.CLEAN_TESTING_CAMPAIGNS_LOGINS)
            .getOrDefault(setOf())

        if (logins.isEmpty()) {
            logger.info("Empty login list")
            return
        }

        val campaignIds = selectCampaigns(logins)
        logger.info("Campaigns for deletion (age > $deletionAge days): $campaignIds")
        if (campaignIds.isEmpty()) {
            return
        }

        deleteCampaigns(campaignIds)
    }

    private fun selectCampaigns(logins: Collection<String>): List<Long> {
        val cidBlacklist: Set<Long> = ppcPropertiesSupport
            .get(PpcPropertyNames.CLEAN_TESTING_CAMPAIGNS_CID_BLACKLIST)
            .getOrDefault(setOf())

        val clientIds: List<Long> = shardHelper.getClientIdsByLogins(logins)
            .values.distinct()

        return cleanTestingCampaignsRepository
            .getCampaignsForDeletion(shard, clientIds, cidBlacklist, deletionAge)
    }

    private fun deleteCampaigns(campaignIds: List<Long>) {
        val campaigns: Map<Long, CampaignSimple> = campaignRepository.getCampaignsSimple(shard, campaignIds)

        cleanTestingCampaignsRepository.makeCampaignsReadyForDeletion(shard, campaignIds)

        val clientIds = campaigns.values.map { ClientId.fromLong(it.clientId) }
        val operatorUidByClientId: Map<ClientId, Long> = rbacService.getChiefsByClientIds(clientIds)

        campaigns.values
            .groupBy { it.clientId }
            .forEach { (clientId, clientCampaigns) ->
                val clientCampaignIds = clientCampaigns.map { it.id }
                logger.info("Deleting client $clientId campaigns: $clientCampaignIds")
                val result = campaignService.deleteCampaigns(
                    clientCampaignIds,
                    operatorUidByClientId[ClientId.fromLong(clientId)]!!,
                    ClientId.fromLong(clientId),
                    Applicability.PARTIAL,
                )

                val errors = result.validationResult.flattenErrors()
                if (errors.isEmpty()) {
                    logger.info("Success")
                } else {
                    logger.info("Errors: $errors")
                }

                val successCampaignIds = result.toResultList()
                    .filter { it.isSuccessful }
                    .map { it.result }
                try {
                    logger.info("Deleting briefs for campaigns: $successCampaignIds")
                    grutApiService.briefGrutApi.deleteObjects(successCampaignIds, ignoreNonexistent = true)
                } catch (e: RuntimeException) {
                    logger.error("Failed to delete briefs", e)
                }
            }
    }
}
