package ru.yandex.direct.jobs.campaign;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesService;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.RECALCULATE_CAMPAIGNS_STATUS_JOB_DEPTH;
import static ru.yandex.direct.common.db.PpcPropertyNames.RECALCULATE_CAMPAIGNS_STATUS_JOB_ITERATION_LIMIT;
import static ru.yandex.direct.common.db.PpcPropertyNames.RECALCULATE_CAMPAIGNS_STATUS_JOB_SLEEP_COEFFICIENT;
import static ru.yandex.direct.common.db.PpcPropertyNames.lastProcessedCampaignId;
import static ru.yandex.direct.common.db.PpcPropertyNames.recalculateCampaignsStatusJobEnabled;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;

/**
 * Контролирует процесс пересчета агрегированных статусов кампаний для тех шардов, для которых активен проперти
 * (@value RECALCULATE_CAMPAIGNS_STATUS_JOB_ENABLED).
 * Параметры работы таска настраиваются во внутреннем инструменте (@link ManageRecalculateCampaignsStatusJobTool) и
 * позволяют выбрать действие и шарды, на которых оно будет применено. Доступные действия -  остановить пересчет,
 * начать пересчет заново или продолжить с последнего обновленного campaignId.
 * В случае ошибки при пересчете id всех кампаний в текущей итерации логируются, а процесс продолжается для
 * следующего набора кампаний.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20), needCheck = ProductionOnly.class)
@Hourglass(periodInSeconds = 300, needSchedule = TypicalEnvironment.class)
public class RecalculateCampaignsStatusJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(RecalculateCampaignsStatusJob.class);

    private static final int DEFAULT_CAMPAIGNS_PER_ITERATION = 100;
    private static final double DEFAULT_SLEEP_COEFFICIENT = 0.5;
    private static final String DEFAULT_RECALCULATION_DEPTH = RecalculationDepthEnum.ALL.value();

    private final PpcPropertiesSupport ppcPropertiesSupport;

    private final AggregatedStatusesService aggregatedStatusesService;
    private final CampaignRepository campaignRepository;

    private final PpcProperty<Double> sleepCoefficientProperty;
    private final PpcProperty<Integer> campaignsPerIterationProperty;
    private final PpcProperty<String> recalculationDepthProperty;

    private static LocalDateTime updateBefore;
    private static int campaignsPerIteration;
    private static RecalculationDepthEnum recalculationDepth;

    // Параметр для остановки джобы после завершения текущего цикла
    private volatile boolean isShutdown = false;

    @Autowired
    public RecalculateCampaignsStatusJob(PpcPropertiesSupport ppcPropertiesSupport,
                                         AggregatedStatusesService aggregatedStatusesService,
                                         CampaignRepository campaignRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        this.aggregatedStatusesService = aggregatedStatusesService;
        this.campaignRepository = campaignRepository;

        sleepCoefficientProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_SLEEP_COEFFICIENT,
                Duration.ofSeconds(20));
        campaignsPerIterationProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_ITERATION_LIMIT);
        recalculationDepthProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_DEPTH);
    }

    /**
     * Конструктор для тестов
     */
    RecalculateCampaignsStatusJob(int shard, PpcPropertiesSupport ppcPropertiesSupport,
                                  AggregatedStatusesService aggregatedStatusesService,
                                  CampaignRepository campaignRepository) {
        super(shard);
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        this.aggregatedStatusesService = aggregatedStatusesService;
        this.campaignRepository = campaignRepository;

        sleepCoefficientProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_SLEEP_COEFFICIENT,
                Duration.ofSeconds(20));
        campaignsPerIterationProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_ITERATION_LIMIT);
        recalculationDepthProperty = ppcPropertiesSupport.get(RECALCULATE_CAMPAIGNS_STATUS_JOB_DEPTH);
    }

    @Override
    public void execute() {
        execute(LocalDateTime.now());
    }

    /**
     * execute для работы тестов с кастомным параметром запуска по времени
     */
    public void execute(LocalDateTime updateBeforeTime) {
        updateBefore = updateBeforeTime;
        int shard = getShard();
        var jobEnabledProperty = recalculateCampaignsStatusJobEnabled(shard);
        var lastProcessedCampaignIdProperty = lastProcessedCampaignId(shard);
        // Эти проперти не будут перечитываться в течение работы джобы
        campaignsPerIteration = campaignsPerIterationProperty.getOrDefault(DEFAULT_CAMPAIGNS_PER_ITERATION);
        var recalculationDepthString = recalculationDepthProperty.getOrDefault(DEFAULT_RECALCULATION_DEPTH);
        try {
            recalculationDepth = RecalculationDepthEnum.fromValue(recalculationDepthString);
        } catch (IllegalArgumentException e) {
            logger.error("Aggregated statuses processing failed to start: ", e);
            return;
        }

        boolean finishedProcessingCampaigns;
        do {
            var jobEnabled = ppcPropertiesSupport.get(jobEnabledProperty).getOrDefault(false);
            if (!jobEnabled) {
                return;
            }
            var sleepCoefficient = sleepCoefficientProperty.getOrDefault(DEFAULT_SLEEP_COEFFICIENT);

            finishedProcessingCampaigns = new RelaxedWorker(sleepCoefficient).callAndRelax(() ->
                    executeRecalculationIteration(shard, jobEnabledProperty.getName(),
                            lastProcessedCampaignIdProperty));
        } while (!finishedProcessingCampaigns && !isShutdown);
    }

    private boolean executeRecalculationIteration(int shard, String jobEnabledPropertyName,
                                                  PpcPropertyName<Long> lastProcessedCampaignIdProperty) {
        // getOrDefault использовать не можем, ибо в процессе вычисления getFirstCampaignId переприсваивается
        // значение проперти + вычислять каждый раз выйдет затратнее по времени, поэтому дважды проверяем на null
        var lastProcessedCampaignId = ppcPropertiesSupport.get(lastProcessedCampaignIdProperty).get();
        var lastProcessedCampaignIdPropertyName = lastProcessedCampaignIdProperty.getName();
        if (lastProcessedCampaignId == null) {
            lastProcessedCampaignId = getFirstCampaignId(shard, lastProcessedCampaignIdPropertyName);
            if (lastProcessedCampaignId == null) {
                return true;
            }
        }

        logger.info("Iteration parameters: lastProcessedCampaignId - {} (shard {})", lastProcessedCampaignId, shard);

        var campaignIds = campaignRepository.getCampaignIdsForStatusRecalculation(shard, lastProcessedCampaignId,
                campaignsPerIteration);
        if (!campaignIds.isEmpty()) {
            recalculateCampaignsStatuses(shard, campaignIds);
        }

        if (campaignIds.size() == campaignsPerIteration) {
            var newLastProcessedCampaignId = campaignIds.get(campaignIds.size() - 1);
            logger.info("Update lastProcessedCampaignId value {} -> {} (shard {})",
                    lastProcessedCampaignId, newLastProcessedCampaignId, shard);
            ppcPropertiesSupport.set(lastProcessedCampaignIdPropertyName, newLastProcessedCampaignId.toString());
            return false;
        } else {
            logger.info("Finished processing campaigns (shard {})", shard);
            ppcPropertiesSupport.set(jobEnabledPropertyName, String.valueOf(false));
            ppcPropertiesSupport.remove(lastProcessedCampaignIdPropertyName);
            return true;
        }
    }

    private Long getFirstCampaignId(int shard, String lastProcessedCampaignIdPropertyName) {
        var newLastProcessedCampaignId = campaignRepository.getNewLastProcessedCampaignId(shard);
        if (newLastProcessedCampaignId == null || newLastProcessedCampaignId <= ID_NOT_SET) {
            logger.warn("No suitable lastProcessedCampaignId value " +
                    "(possibly there are no campaigns in this shard)");
            return null;
        } else {
            logger.info("New lastProcessedCampaignId {} (shard {})", newLastProcessedCampaignId, shard);
            ppcPropertiesSupport.set(lastProcessedCampaignIdPropertyName, newLastProcessedCampaignId.toString());
            return newLastProcessedCampaignId;
        }
    }

    private void recalculateCampaignsStatuses(int shard, Collection<Long> campaignIds) {
        try {
            aggregatedStatusesService.fullyRecalculateStatuses(shard, updateBefore, new HashSet<>(campaignIds),
                    recalculationDepth);
        } catch (RuntimeException e) {
            logger.error("Aggregated statuses processing failed for cids: {}", campaignIds, e);
        }
    }

    /**
     * Останавливает исполнение джобы после завершения текущего цикла
     */
    @Override
    public void onShutdown() {
        isShutdown = true;
        logger.info("Job will shutdown after the current cycle");
    }
}
